/* $Id: query.c,v 1.19 2005/12/14 08:22:32 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Postgres Query Interface
 */
#include <postgres.h>
#include <miscadmin.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/catversion.h>
#include <executor/executor.h>
#include <executor/execdesc.h>
#include <executor/tstoreReceiver.h>
#include <nodes/params.h>
#include <parser/analyze.h>
#include <tcop/tcopprot.h>
#include <tcop/dest.h>
#include <tcop/pquery.h>
#include <tcop/utility.h>
#include <utils/memutils.h>
#include <utils/array.h>
#include <utils/palloc.h>
#include <utils/portal.h>
#include <utils/relcache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <rewrite/rewriteHandler.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/error.h>

#include <pypg/query.h>
#include <pypg/call.h>
#include <pypg/call/portal.h>

PyObj
PyList_FromTupleDescAndTuplestore(TupleDesc td, Tuplestorestate *ts)
{
	PyObj tdo, hto, rob;
	bool should_free;
	HeapTuple ht;

	tdo = PyPgTupleDesc_New(td);
	if (tdo == NULL) return(NULL);

	rob = PyList_New(0);
	if (rob == NULL) goto decref_tdo;

	while ((ht = (HeapTuple) tuplestore_gettuple(ts, true, &should_free)))
	{
		hto = PyPgHeapTuple_New(tdo, ht);
		if (should_free) heap_freetuple(ht);
		if (hto == NULL) goto decref_rob;

		PyList_Append(rob, hto);
		Py_DECREF(hto);
		if (PyErr_Occurred()) goto decref_rob;
	}
	Py_DECREF(tdo);

	return(rob);
decref_rob:
	Py_DECREF(rob);
decref_tdo:
	Py_DECREF(tdo);
	return(NULL);
}

ParamListInfo
ParamListInfo_FromTupleDescAndHeapTuple(TupleDesc td, HeapTuple ht)
{
	unsigned int i, natts = td->natts;
	ParamListInfo pli;

	pli = palloc0((natts + 1) * sizeof(ParamListInfoData));
	for (i = 0; i < natts; ++i)
	{
		pli[i].kind = PARAM_NUM;
		pli[i].name = NameStr(td->attrs[i]->attname);
		pli[i].id = i + 1;
		pli[i].ptype = td->attrs[i]->atttypid;

		pli[i].value = fastgetattr(ht, i + 1, td, &(pli[i].isnull));
	}
	pli[natts].kind = PARAM_INVALID;

	return(pli);
}

static PyMemberDef PyPgQuery_Members[] = {
	{"lastoid", T_UINT, offsetof(struct PyPgQuery, q_lastoid), RO,
		PyDoc_STR("oid returned from last utility statement")},
	{"arguments", T_OBJECT, offsetof(struct PyPgQuery, q_input), RO,
		PyDoc_STR("Deprecated, see the input attribute")},
	{"input", T_OBJECT, offsetof(struct PyPgQuery, q_input), RO,
		PyDoc_STR("A Postgres.TupleDesc of the query's parameter types")},
	{"output", T_OBJECT, offsetof(struct PyPgQuery, q_output), RO,
		PyDoc_STR("A Postgres.TupleDesc of the query's result tuple")},
	{"source", T_OBJECT, offsetof(struct PyPgQuery, q_source), RO,
		PyDoc_STR("The source that defines this query")},
	{NULL}
};

/*
 * execute
 * 	A simple function for non-SELECT, non-Utility statements.
 * 	All result producing SELECT statements are portalized.
 */
static PyObj
execute(PyObj self, ParamListInfo pli, Query *query, Plan *plan)
{
	long processed;
	Oid lastoid;
	PyObj rob = NULL;
	QueryDesc *qd = NULL;

	qd = CreateQueryDesc(query, plan,
		GetLatestSnapshot(), GetLatestSnapshot(),
		None_Receiver, pli, false);

	ExecutorStart(qd, false);
	ExecutorRun(qd, ForwardScanDirection, 0);

	lastoid = qd->estate->es_lastoid;
	processed = qd->estate->es_processed;

	ExecutorEnd(qd);
	CommandCounterIncrement();

	PyPgQuery_FixLastOid(self, lastoid);
	rob = PyLong_FromUnsignedLong(processed);
	return(rob);
}

static PyObj
utility(Node *utilstmt)
{
	TupleDesc utd;
	PyObj rob = NULL;

	utd = UtilityTupleDescriptor(utilstmt);
	if (utd)
	{
		Tuplestorestate *ts;
		DestReceiver *tsr;

		ts = tuplestore_begin_heap(false, false, work_mem);
		tsr = CreateTuplestoreDestReceiver(ts, CurrentMemoryContext);

		ProcessUtility(utilstmt, NULL, tsr, NULL);
		
		rob = PyList_FromTupleDescAndTuplestore(utd, ts);

		tsr->rDestroy(tsr);
		tuplestore_end(ts);
	}
	else
	{
		ProcessUtility(utilstmt, NULL, None_Receiver, NULL);
		rob = Py_None;
		Py_INCREF(Py_None);
	}

	CommandCounterIncrement();

	return(rob);
}

static PyMethodDef PyPgQuery_Methods[] = {
	/*
	{"prepare", (PyCFunction) query_prepare, METH_NOARGS,
		PyDoc_STR("explicitly prepare the query")},
	*/
	{NULL}
};

static void
query_dealloc(PyObj self)
{
	PyObj ob;

	PyPgQuery_FixLastOid(self, InvalidOid);

	ob = PyPgQuery_FetchSource(self);
	PyPgQuery_FixSource(self, NULL);
	Py_DECREF(ob);

	ob = PyPgQuery_FetchInput(self);
	PyPgQuery_FixInput(self, NULL);
	Py_DECREF(ob);

	ob = PyPgQuery_FetchOutput(self);
	PyPgQuery_FixOutput(self, NULL);
	Py_DECREF(ob);

	MemoryContextDelete(PyPgQuery_FetchExecutionMemoryContext(self));
	MemoryContextDelete(PyPgQuery_FetchMemoryContext(self));
	PyPgQuery_FixExecutionMemoryContext(self, NULL);
	PyPgQuery_FixMemoryContext(self, NULL);

	self->ob_type->tp_free(self);
}

static PyObj
query_repr(PyObj self)
{
	PyObj argrepr, rob;
	argrepr = PyObject_Repr(PyPgQuery_FetchInput(self));
	rob = PyString_FromFormat(
		"<Postgres.Query%s>",
		PyString_AS_STRING(argrepr)
	);
	Py_DECREF(argrepr);
	return(rob);
}

static PyObj
query_call(PyObj self, PyObj args, PyObj kw)
{
	MemoryContext former;
	int argc, argtypec;
	PyObj rob = NULL;

	/* Query Parameters */
	TupleDesc p_types;
	HeapTuple p_tuple = NULL;

	ListCell *queryc, *planc;
	List *queryl, *planl;

	argc = PySequence_Length(args);
	p_types = PyPgTupleDesc_FetchTupleDesc(PyPgQuery_FetchInput(self));
	argtypec = p_types->natts;

	if (argc != argtypec)
	{
		PyErr_Format(PyExc_TypeError,
			"query requires %d arguments, given %d", argtypec, argc
		);
		return(NULL);
	}

	Assert(p_types != NULL);

	former = CurrentMemoryContext;
	PG_TRY();
	{
		Query *query;
		Plan *plan;

		MemoryContextSwitchTo(PyPgQuery_FetchExecutionMemoryContext(self));
		if (argc > 0)
		{
			p_tuple = HeapTuple_FromTupleDescAndIterable(p_types, args);
			if (p_tuple == NULL)
				goto tryend;
		}

		queryl = PyPgQuery_FetchQueryList(self);
		planl = PyPgQuery_FetchPlanList(self);
		forboth(queryc, queryl, planc, planl)
		{
			query = (Query *) lfirst(queryc);
			plan = (Plan *) lfirst(planc);

			if (query->commandType == CMD_SELECT && query->into == NULL)
			{
				PyObj p_tuple_ob;

				if (p_tuple)
				{
					p_tuple_ob =
						PyPgHeapTuple_New(PyPgQuery_FetchInput(self), p_tuple);
					if (p_tuple_ob == NULL) goto tryend;
				}
				else
					p_tuple_ob = EmptyPyPgHeapTuple;

				rob = PyPgPortal_New(self, p_tuple_ob);
			}
			else if (query->commandType == CMD_UTILITY)
			{
				rob = utility(query->utilityStmt);
			}
			else
			{
				ParamListInfo pli = NULL;

				if (argc > 0)
					pli = ParamListInfo_FromTupleDescAndHeapTuple(p_types, p_tuple);
				rob = execute(self, pli, query, plan);
			}
		}
tryend:;
	}
	PYPG_CATCH_END();

	former = MemoryContextSwitchTo(former);
	MemoryContextReset(PyPgQuery_FetchExecutionMemoryContext(self));

	return(rob);
}

static PyObj
query_iter(PyObj self)
{
	PyObj args, rob;
	args = PyTuple_New(0);
	rob = query_call(self, args, NULL);
	Py_DECREF(args);
	return(rob);
}

static PyObj
query_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj source = NULL, rob;

	if (!PyPgArg_ParseTypeSource(args, kw, &source))
		return(NULL);

	rob = PyPgQuery_Initialize(subtype->tp_alloc(subtype, 0), source);
	return(rob);
}

PyDoc_STRVAR(query_doc, "Postgres Query");
PyTypeObject PyPgQuery_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Query",					/* tp_name */
	sizeof(struct PyPgQuery),		/* tp_basicsize */
	0,										/* tp_itemsize */
	query_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	query_repr,							/* tp_repr */
	NULL,									/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	query_call,							/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	query_doc,							/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	query_iter,							/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgQuery_Methods,				/* tp_methods */
	PyPgQuery_Members,				/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	query_new,							/* tp_new */
};

PyObj
PyPgQuery_Initialize(PyObj self, PyObj source)
{
	MemoryContext former, q_context = NULL, q_exec_context = NULL;
	PyObj source_str, input = NULL, output = NULL;

	if (self == NULL) return(NULL);

	PyPgQuery_FixSource(self, source);
	PyPgQuery_FixLastOid(self, InvalidOid);
	PyPgQuery_FixInput(self, NULL);
	PyPgQuery_FixRawParseTree(self, NULL);

	source_str = PyObject_Str(source);
	if (source_str == NULL)
	{
		self->ob_type->tp_free(self);
		return(NULL);
	}

	former = CurrentMemoryContext;
	PG_TRY();
	{
		Node *rpt = NULL;
		ListCell * volatile qli;
		List * volatile rqlist;
		List * volatile qlist = NIL;
		List * volatile plist = NIL;

		q_context = AllocSetContextCreate(
			PythonMemoryContext,
			"Python Permanent Query Context",
			ALLOCSET_DEFAULT_MINSIZE,
			ALLOCSET_DEFAULT_INITSIZE,
			ALLOCSET_DEFAULT_MAXSIZE
		);
		PyPgQuery_FixMemoryContext(self, q_context);
		MemoryContextSwitchTo(q_context);

		q_exec_context = AllocSetContextCreate(
			q_context,
			"Python Execution Query Context",
			ALLOCSET_DEFAULT_MINSIZE,
			ALLOCSET_DEFAULT_INITSIZE,
			ALLOCSET_DEFAULT_MAXSIZE
		);
		PyPgQuery_FixExecutionMemoryContext(self, q_exec_context);
		MemoryContextSwitchTo(q_exec_context);

		{
			List *ptl;

			MemoryContextSwitchTo(q_exec_context);

			ptl = pg_parse_query(PyString_AS_STRING(source_str));
			if (list_length(ptl) > 1)
				ereport(ERROR, (
					errcode(ERRCODE_SYNTAX_ERROR),
					errmsg(
						"cannot insert multiple commands into a prepared statement"
					),
					errhint(
						"The Execute method may be used to run multiple statements."
					)
				));
			rpt = linitial(ptl);

			if (IsA(rpt, TransactionStmt))
			{
				ereport(ERROR, (
					errcode(ERRCODE_S_R_E_PROHIBITED_SQL_STATEMENT_ATTEMPTED),
					errmsg("forbidden transaction statement in Query string"),
					errhint(
						"Use the Transact function or the Transaction context "
						"to manage transactions."
					)
				));
			}
			if (IsA(rpt, CopyStmt) && ((CopyStmt *) rpt)->filename == NULL)
			{
				ereport(ERROR, (
					errcode(ERRCODE_S_R_E_PROHIBITED_SQL_STATEMENT_ATTEMPTED),
					errmsg("forbidden COPY TO or FROM in Query string"),
					errhint("Standard I/O copies are prohibited at this level.")
				));
			}

			MemoryContextSwitchTo(q_context);
			rpt = copyObject(rpt);
			PyPgQuery_FixRawParseTree(self, rpt);
		}

		MemoryContextSwitchTo(q_exec_context);
		{
			int i, nargs = 0;
			Oid *argtype_oids = NULL;
			TupleDesc input_td;

			rqlist = parse_analyze_varparams(rpt, &argtype_oids, &nargs);
			for (i = 0; i < nargs; ++i)
			{
				Oid at = argtype_oids[i];

				if (at == InvalidOid || at == UNKNOWNOID)
				{
					ereport(ERROR, (
						errcode(ERRCODE_INDETERMINATE_DATATYPE),
						errmsg("could not determine data type of parameter $%d",
							i + 1)
					));
				}
			}

			input_td = TupleDesc_FromNamesAndOids(nargs, NULL, argtype_oids);
			if (input_td == NULL)
			{
				ereport(ERROR, (
					errmsg("failed to create TupleDesc for query parameters")
				));
			}
			input = PyPgTupleDesc_New(input_td);
			if (input == NULL)
			{
				ereport(ERROR, (
					errmsg("failed to create TupleDesc object for query parameters")
				));
			}
			PyPgQuery_FixInput(self, input);

			foreach(qli, rqlist)
			{
				Query *qt = (Query *) lfirst(qli);

				if (qt->commandType == CMD_UTILITY)
				{
					qlist = lappend(qlist, qt);
				}
				else
				{
					List *rewritten = QueryRewrite(qt);
					qlist = list_concat(qlist, rewritten);
				}
			}
		}

		switch (ChoosePortalStrategy(qlist))
		{
			case PORTAL_ONE_SELECT:
			{
				Query *q;
				TupleDesc output_td;
				q = (Query *) linitial(qlist);
				output_td = ExecCleanTypeFromTL(q->targetList, false);
				output = PyPgTupleDesc_New(output_td);
			}
			break;

			case PORTAL_UTIL_SELECT:
			{
				Query *q;
				TupleDesc output_td;
				q = (Query *) linitial(qlist);
				output_td = UtilityTupleDescriptor(q->utilityStmt);
				output = PyPgTupleDesc_New(output_td);
			}
			break;

			default:
			{
				output = Py_None;
				Py_INCREF(output);
			}
			break;
		}
		PyPgQuery_FixOutput(self, output);

		foreach(qli, qlist)
		{
			Query * volatile qtree = (Query *) lfirst(qli);
			Plan * volatile planTree;

			planTree = pg_plan_query(qtree, NULL);
			plist = lappend(plist, planTree);
		}

		MemoryContextSwitchTo(q_context);
		qlist = copyObject(qlist);
		plist = copyObject(plist);
		PyPgQuery_FixQueryList(self, qlist);
		PyPgQuery_FixPlanList(self, plist);
	}
	PYPG_CATCH_END();
	MemoryContextReset(q_exec_context);

	MemoryContextSwitchTo(former);
	Py_DECREF(source_str);

	if (PyErr_Occurred())
	{
		Py_XDECREF(input);
		Py_XDECREF(output);
		self->ob_type->tp_free(self);
		MemoryContextDelete(q_exec_context);
		MemoryContextDelete(q_context);
		return(NULL);
	}

	Py_INCREF(source);

	return(self);
}
/*
 * vim: ts=3:sw=3:noet:
 */
