#!/bin/sh

# The purpose of this script is to generate all needed Makefile.am
# and to update the AC_OUTPUT instruction in the configure.ac files.
# It scans directories recursively.

################################
## Script options - Beginning ##
################################

# AUTOMAKE_OPTIONS option in Makefile.am files
AUTOMAKE_OPTIONS="foreign"

# Extensions to add to the distribution in ./files & ./doc
EXTENSIONS1=( "*.txt" "*.h" "*.c" "*.sh" "*.pgs" "*.out" "*.html" "*.xml" "*.css" )
# Extensions to add to the distribution in ./lib
EXTENSIONS2=( "*.hh" "*.ll" "*.h" "*.yy" "*.sh" "*.bat" "*.h.in" )
# Extensions to add to the distribution in ./test
EXTENSIONS3=( "*.h" )
# There is no parameter of this type for ./src because Makefile.am is not
# generated automatically.

# Library name (./lib)
LIB_NAME="libpgs"
# Test program name (./test)
TST_NAME="pgsTest"
# There is no parameter of this type for ./src because Makefile.am is not
# generated automatically.

# Directory that contains headers (no ./ & relative to ./lib)
INCLUDEDIR="include"

##########################
## Script options - End ##
##########################

###########################
## Utilities - Beginning ##
###########################

function scan_directory
{
	for dir in `find "$1" -type d`; do
		# Ignore CVS directories
		if test `expr match "$dir" '.*CVS.*'` -eq 0
		then
			if test "${dir:2}" != ""
			then
				MAKEFILE_PATHS="${dir:2}/Makefile $MAKEFILE_PATHS"
			else
				MAKEFILE_PATHS="Makefile $MAKEFILE_PATHS"
			fi
			echo -n "Exploring $dir... "
			SUBDIRS=""
			EXTRA_DIST=""
			# Add immediate subdirectories to the SUBDIRS list
			for sub in `find "$dir" -maxdepth 1 -type d`; do
				if test $sub != $dir -a `expr match "$sub" '.*CVS.*'` -eq 0
				then
					SUBDIR=${sub##.*\/}
					SUBDIRS="$SUBDIR $SUBDIRS"
				fi
			done
			# Extensions to include as ESTRA_DIST files
			for ext in "${EXTENSIONS[@]}"
			do
				for sub in `find "$dir" -maxdepth 1 -name "$ext"`; do
					EXTRA_FILE=${sub##.*\/}
					EXTRA_DIST="$EXTRA_FILE $EXTRA_DIST"
				done
			done
			# Write Makefile.am in the current explored directory
			echo "AUTOMAKE_OPTIONS = $AUTOMAKE_OPTIONS" > $dir/Makefile.am
			echo "EXTRA_DIST = $EXTRA_DIST" >> $dir/Makefile.am
			echo "" >> $dir/Makefile.am
			echo "SUBDIRS = $SUBDIRS" >> $dir/Makefile.am
			echo "done"
		fi
	done	
}

function assign_extensions
{
	i=0
	if test "$1" == "1"
	then
		for ext in "${EXTENSIONS1[@]}"
		do
			EXTENSIONS[$i]="$ext"
			i=`expr $i + 1`
		done
	elif test "$1" == "2"
	then
		for ext in "${EXTENSIONS2[@]}"
		do
			EXTENSIONS[$i]="$ext"
			i=`expr $i + 1`
		done
	else
		for ext in "${EXTENSIONS3[@]}"
		do
			EXTENSIONS[$i]="$ext"
			i=`expr $i + 1`
		done
	fi
}

#####################
## Utilities - End ##
#####################

echo "This script builds Makefile.am files and updates configure.ac file."
echo "Usage: $0"

################################################################################

echo ""
echo "Processing root"
echo ""

MAKEFILE_PATHS=""
assign_extensions 1
BIN_PROGRAMS=""

# Go through each directory
scan_directory "./files"
scan_directory "./doc"

# Write top Makefile.am
echo -n "Writing top Makefile.am... "
echo "AUTOMAKE_OPTIONS = $AUTOMAKE_OPTIONS" > Makefile.am
echo "EXTRA_DIST = autoclean.sh autofind.sh autogen.sh windows.sh" >> Makefile.am
echo "" >> Makefile.am
echo "DIST_SUBDIRS = doc files lib src test" >> Makefile.am
echo "" >> Makefile.am
echo "if ENABLE_TEST" >> Makefile.am
echo "PGS_PGM = test" >> Makefile.am
echo "else" >> Makefile.am
echo "PGS_PGM = src" >> Makefile.am
echo "endif" >> Makefile.am
echo 'SUBDIRS = files lib $(PGS_PGM)' >> Makefile.am
echo "" >> Makefile.am
echo "done"

echo -n "Updating configure.ac... "
CONFIGURE=`cat configure.ac | sed -e "s#AC_OUTPUT.*#AC_OUTPUT([ Makefile $MAKEFILE_PATHS ])#g"`
echo "$CONFIGURE" > configure.ac
echo "done"

echo ""
echo "Leaving root"
echo ""

################################################################################

echo ""
echo "Processing lib"
cd lib
echo ""

MAKEFILE_PATHS=""
assign_extensions 2
BIN_PROGRAMS="$LIB_NAME"

# Go through each directory
scan_directory "./"

# Get sources
PROGRAM_SOURCES=""
echo -n "Finding program sources... "
for src in `find ./ -name "*.c[pc]*"`; do
	src=${src:2}
	PROGRAM_SOURCES="$src $PROGRAM_SOURCES"
done
echo "done"

# Get headers
PROGRAM_HEADERS=""
echo -n "Finding program headers... "
for src in `find "./$INCLUDEDIR" -name "*.h*"`; do
	src=${src:2}
	PROGRAM_HEADERS="$src $PROGRAM_HEADERS"
done
echo "done"

# Get directories
DIRECTORIES=""
echo -n "Finding directories... "
for dir in `find ./ -maxdepth 1 -type d`; do
	DIRECTORIES="${dir##.*\/} $DIRECTORIES"
done
echo "done"

# Write top Makefile.am
echo -n "Writing top Makefile.am... "
echo "" >> Makefile.am
echo "if INSTLIB" >> Makefile.am
echo "lib_LIBRARIES = $BIN_PROGRAMS.a" >> Makefile.am
echo "else" >> Makefile.am
echo "noinst_LIBRARIES = $BIN_PROGRAMS.a" >> Makefile.am
echo "endif" >> Makefile.am
echo "" >> Makefile.am
echo "$BIN_PROGRAMS""_a_SOURCES = $PROGRAM_SOURCES" >> Makefile.am
echo "" >> Makefile.am
echo "done"

# Complete include/Makefile.am
PROGRAM_HEADERS=`echo "$PROGRAM_HEADERS" | sed -e "s#$INCLUDEDIR/##g"`
echo "" >> $INCLUDEDIR/Makefile.am
echo "if INSTLIB" >> $INCLUDEDIR/Makefile.am
echo "nobase_include_HEADERS = $PROGRAM_HEADERS" >> $INCLUDEDIR/Makefile.am
echo "endif" >> $INCLUDEDIR/Makefile.am
echo "" >> $INCLUDEDIR/Makefile.am

echo -n "Updating configure.ac... "
CONFIGURE=`cat configure.ac | sed -e "s#AC_OUTPUT.*#AC_OUTPUT([ $MAKEFILE_PATHS ])#g"`
echo "$CONFIGURE" > configure.ac
echo "done"

echo ""
echo "Leaving lib"
cd ..
echo ""

################################################################################

echo ""
echo "Processing test"
cd test
echo ""

MAKEFILE_PATHS=""
assign_extensions 2
BIN_PROGRAMS="$TST_NAME"

# Go through each directory
scan_directory "./"

# Get sources
PROGRAM_SOURCES=""
echo -n "Finding program sources... "
for src in `find ./ -name "*.c[pc]*"`; do
	src=${src:2}
	PROGRAM_SOURCES="$src $PROGRAM_SOURCES"
done
echo "done"

# Write top Makefile.am
echo -n "Writing top Makefile.am... "
echo "" >> Makefile.am
echo "bin_PROGRAMS = $BIN_PROGRAMS" >> Makefile.am
echo "" >> Makefile.am
echo "$BIN_PROGRAMS""_SOURCES = $PROGRAM_SOURCES" >> Makefile.am
echo "" >> Makefile.am
echo "INCLUDES = -I../lib/include" >> Makefile.am
echo "$BIN_PROGRAMS""_LDADD = ../lib/libpgs.a" >> Makefile.am
echo "" >> Makefile.am
echo "done"

echo -n "Updating configure.ac... "
CONFIGURE=`cat configure.ac | sed -e "s#AC_OUTPUT.*#AC_OUTPUT([ $MAKEFILE_PATHS ])#g"`
echo "$CONFIGURE" > configure.ac
echo "done"

echo ""
echo "Leaving test"
cd ..
echo ""

################################################################################
