//
//  PostgreSQL.m
//  PostgresSQL
//
//  Created by Pascal on Sat Dec 14 2002.
//  Copyright (c) 2002 P3 Consulting. All rights reserved.
//

#import "PostgreSQL.h"
#import "PgSQLBindItem.h"
#import "PgASyncThread.h"
#import "LoginWindowController.h"
#import "LoginPanelController.h"
#import "PostgreSQLPrivate.h"
#import "PgSQLResult.h"
#import <objc/objc-runtime.h>
#import "pgtypes.h"

#import <stdio.h>
#import <unistd.h>

#define CLEAR_LASTRESULT 		if (_lastResult != nil) { PQclear(_lastResult) ; _lastResult = nil	;} _curRowIndex = -1 ; _rowsInResult = 0; _fieldsInResult = 0
#define	LASTRESULT_ISOK			((PQresultStatus(_lastResult) == PGRES_COMMAND_OK) || (PQresultStatus(_lastResult) == PGRES_TUPLES_OK))
#define	RESULT_ISOK(result)			((PQresultStatus(result) == PGRES_COMMAND_OK) || (PQresultStatus(result) == PGRES_TUPLES_OK))

NSString *PostgreSQLNotification = @"PostgreSQLNotification";
NSString *PostgreSQLResultAvailable = @"PostgreSQLResultAvailable";
NSString *PostgreSQLTimeOut = @"PostgreSQLTimeOut";
NSString *PgSQLuserInfoNotificationField = @"userInfo"	;
NSString *PgSQLconditionNotificationField = @"condition"	;
NSString *PgSQLconnectionNotificationField = @"PostgreSQL"	;

static NSString *nilConnectionMsg = @"_conn is nil";
static NSString *nilBindingArray = @"_bindingArray is nil";
static NSString *nilLastResultMsg = @"_lastResult is nil";
static NSString *nilArgumentMsg = @"nil argument";
static NSString *bindTypeError = @"bind type doesn't correspond to Postgre type";

NSString *invalidRowIndex = @"current row index is invalid";

#if	!DEMO_MODE
static NSString *noTransactionMsg = @"no transcation opened";
#endif

NSString	*PgSQLColumnsKeyField = @"columns"	;
	NSString	*PgSQLNameKeyField = @"name"	;
	NSString	*PgSQLTypeKeyField = @"typeOid"	;
	NSString	*PgSQLTypeNameKeyField = @"typeName"	;
NSString	*PgSQLRowsKeyField = @"rows"	;

@implementation PostgreSQL

static BOOL	gDebugMode = NO	;

/*
static	NSMutableDictionary	*_postgreSQLConnections	= nil	;

void
handle_sigint(SIGNAL_ARGS)
{
	NSEnumerator	*aEnum = [_postgreSQLConnections objectEnumerator]	;
	PostgreSQL	*_conn	;
	
		while ((_conn = [aEnum nextObject]) != nil)	{
			[_conn disconnect]	;
		}
}

+ (void)load
{
	sigset_t	my_sigs	;
	struct sigaction	sa_new, sa_old	;

		_postgreSQLConnections = [[NSMutableDictionary dictionaryWithCapacity:10] retain]	;
		sigemptyset(&my_sigs)	;
		sigaddset(&my_sigs,SIGINT)	;
		sa_new.sa_handler = handle_sigint	;
		sigemptyset(&sa_new.sa_mask)	;
		sa_new.sa_flags = 0	;
		sigaction(SIGINT,&sa_new,&sa_old)	;
}
*/

+ (void)setDebug:(BOOL)yn
{
		gDebugMode = yn	;
}

+ (void)initialize
{
		[PostgreSQL setVersion:1]	;
}

static NSStringEncoding postgreEncoding2Mac(NSString *encoding)
{
			if ([encoding isEqualToString:@"'SQL_ASCII'"]) {
				return NSASCIIStringEncoding;
			}
			if ([encoding isEqualToString:@"'UNICODE'"]) {
				return NSUnicodeStringEncoding;
			}
			if ([encoding isEqualToString:@"'LATIN1'"]) {
				return NSISOLatin1StringEncoding;
			}
			if ([encoding isEqualToString:@"'LATIN2'"]) {
				return NSISOLatin2StringEncoding;
			}
			if ([encoding isEqualToString:@"'WIN'"]) {
				return NSWindowsCP1251StringEncoding;
			}
			if ([encoding isEqualToString:@"'EUC_JP'"]) {
				return NSJapaneseEUCStringEncoding;
			}
			if ([encoding isEqualToString:@"'SJIS'"]) {
				return NSShiftJISStringEncoding;
			}
			if ([encoding isEqualToString:@"'WIN1250'"]) {
				return NSWindowsCP1250StringEncoding;
			}

			if ([encoding isEqualToString:@"'EUC_CN'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingEUC_CN);
			}
			if ([encoding isEqualToString:@"'EUC_KR'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingEUC_KR);
			}
			if ([encoding isEqualToString:@"'EUC_TW'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingEUC_TW);
			}

			if ([encoding isEqualToString:@"'LATIN3'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISOLatin3);
			}
			if ([encoding isEqualToString:@"'LATIN4'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISOLatin4);
			}
			if ([encoding isEqualToString:@"'LATIN5'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISOLatin5);
			}
			if ([encoding isEqualToString:@"'LATIN6'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISOLatin6);
			}
			if ([encoding isEqualToString:@"'LATIN7'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISOLatin7);
			}
			if ([encoding isEqualToString:@"'LATIN8'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISOLatin8);
			}
			if ([encoding isEqualToString:@"'LATIN9'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISOLatin9);
			}
			if ([encoding isEqualToString:@"'KOI8'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingKOI8_R);
			}
			if ([encoding isEqualToString:@"'BIG5'"]) {
				return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingBig5);
			}
	
			return NSISOLatin1StringEncoding;
}

- (NSStringEncoding) showClientEncoding
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;

	NSDictionary *vars = [self getVariables]	;
	
		if (vars == nil)
			return nil	;
			
	NSString		*encoding = [vars objectForKey:@"Current client encoding"]	;
	
		if (encoding != nil)	{
			return postgreEncoding2Mac(encoding)	;
		}
	
		return NSISOLatin1StringEncoding;
}

- (NSStringEncoding) showServerEncoding
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;

	NSDictionary	*vars = [self getVariables]	;
	
		if (vars == nil)
			return nil	;
			
	NSString		*encoding = [vars objectForKey:@"Current server encoding"]	;
	
		if (encoding != nil)	{
			return postgreEncoding2Mac(encoding)	;
		}
	
		return NSISOLatin1StringEncoding;
}

- (void)setDebugMode:(BOOL)turnOn
{
		_debugMode = turnOn	;
}

- (BOOL)debugMode
{
		return _debugMode	;
}

-(BOOL)startTracing:(const char *)traceFileName
{	
		[self stopTracing]	;
		_traceFile = fopen(traceFileName,"w")	;
	
		if (_traceFile != NULL)	{
			PQtrace(_conn, _traceFile)	;
		}
		
		return (_traceFile != NULL)	;
}

-(void)stopTracing
{
		if (_traceFile != NULL)	{
			fclose(_traceFile)	;
			PQuntrace(_conn)	;
		}
}

-(void)handlePgNotify:(NSTimer *)firingTimer
{
	PGresult	*res	;
	PGnotify	*notify	;

		if (_conn == NULL)
			return	;
			
		res = PQexec(_conn," ")	;
		while ((notify = PQnotifies(_conn)) != nil)	{
			PgSQLListener	*listener = [_listeners objectForKey:[NSString stringWithCString:notify->relname]]	;
			
			if (listener != nil)	{	
				NSMutableDictionary	*aDict = [listener userInfo]	;
				
				[aDict setObject:[NSNumber numberWithInt:notify->be_pid] forKey:@"pid"]	;
				[[NSNotificationCenter defaultCenter] postNotificationName:PostgreSQLNotification object:[listener toBeNotified] userInfo:aDict]	;
				
				if ([[listener toBeNotified] respondsToSelector:@selector(postgreSQL:notification:)] )	
					objc_msgSend([listener toBeNotified], @selector(postgreSQL:notification:), self, aDict)	;

				if ( [_delegate respondsToSelector:@selector(postgreSQL:notification:)] )
					objc_msgSend(_delegate, @selector(postgreSQL:notification:), self, aDict)	;
//					[_delegate postgreSQL:self notification:aDict];		
			}
			free(notify)	;
		}
		PQclear(res)	;
}

- (id)init
{
		self = [super init]	;
		_uniqueKeyID = (NSString *)CFUUIDCreateString(kCFAllocatorDefault,CFUUIDCreate(kCFAllocatorDefault))	;
		_conn = nil		;
		_debugMode = NO	;
		_traceFile = NULL	;
		_lastResult = nil	;
		_cmdBuffer = [[NSMutableString string] retain]	;
		_listeners = [[NSMutableDictionary dictionaryWithCapacity:10] retain]	;
		_commandExecuted = NO	;
		_asyncCommandInProgress = NO	;
		_asyncCommandResultAvailable = NO	;
		_bindingArray = nil	;
		_transactionLevel = 0	;
		_timeOut = 10	;
		_listeningTimer = [[NSTimer scheduledTimerWithTimeInterval:1.0 target:self selector:@selector(handlePgNotify:) userInfo:nil repeats:YES] retain];
		
		_asyncThread = nil	;
		_delegate = nil		;
		_localUSDict = [[NSDictionary dictionaryWithObject:@"." forKey:NSDecimalSeparator] retain]	;
		[[NSNotificationCenter defaultCenter] 
			addObserver:self
			selector:@selector(handleAppWillTermNotification:)
			name:NSApplicationWillTerminateNotification
			object:nil]	;
		
		_results = [[NSMutableArray arrayWithCapacity:10] retain]	;
/*
		[_postgreSQLConnections setObject:self forKey:_uniqueKeyID]	;
*/
		
		return self	;
}

-(id)delegate
{
		return _delegate	;
}

-(void)setDelegate:(id)newDelegate
{
		[_delegate release]	;
		_delegate = [newDelegate retain]	;
}

-(void)handleAppWillTermNotification:(NSNotification *)notification
{
		[self disconnect]	;
}

-(NSString *)uniqueID
{
		return _uniqueKeyID	;
}

-(void)_disconnect:(BOOL)forDealloc
{
		if (forDealloc)
			[_listeningTimer invalidate]	;

		if ([_results count] > 0)	{
			[_results release]	;
			_results = [[NSMutableArray arrayWithCapacity:10] retain]	;
		}
		
		if (_conn != nil)	{
			CLEAR_LASTRESULT	;
			PQfinish(_conn)	;
			_conn = nil	;
		}
}

-(void)dealloc
{
		[[NSNotificationCenter defaultCenter]  removeObserver:self]	;

		[self _disconnect:YES]	;

		[_listeningTimer release]	;
		[_bindingArray release];
		[_listeners release]	;
		[_cmdBuffer release]	;
		[_localUSDict release]	;
		[_results release]		;
/*
		[_postgreSQLConnections removeObjectForKey:_uniqueKeyID]	;
*/

		[_uniqueKeyID release]	;
		
		[super dealloc]		;
}

-(void)disconnect
{
		[self _disconnect:NO]	;
}

- (BOOL)connectToDatabase:(const char *)dbName onHost:(const char *)hostName login:(const char *)loginName password:(const char *)password return:(int *)returnCode;
{
	char	connInfo[1024]	;
	
		sprintf(connInfo,"host='%s' dbname='%s' user='%s' password='%s'",hostName,dbName,loginName,password)	;
		
		return [self connectToDatabase:connInfo	return:returnCode]	;
}

- (BOOL)connectToDatabase:(const char *)connectInfo return:(int *)returnCode
{
		_debugMode = gDebugMode	;
		[self _disconnect:NO]	;
		
		_conn = PQconnectdb(connectInfo)	;
		
		*returnCode = PQstatus(_conn)	;
		
		if (*returnCode == CONNECTION_OK)	{
			_asyncConnection = [PgASyncThread startASyncThreadForConnection:self];
			if (_asyncConnection != nil)
				if (PQsetnonblocking(_conn,1) < 0)
					NSLog(@"Error setting non blocking mode %s",PQerrorMessage(_conn))	;
		}
		return (*returnCode == CONNECTION_OK)	;
}

- (BOOL)connectAskingUser:(int *)returnCode forWindow:(NSWindow *)hostWindow
{
		if (hostWindow == nil)	{
			LoginWindowController	*aController	;
			
				aController = [[LoginWindowController alloc] init];
				[aController showWindow:self];
				[aController setConnection:self];
				
				[NSApp runModalForWindow:[aController window]]	;
				
				*returnCode = [aController errorCode]	;
				[aController close];	
				return [aController connectionOK]	;
		}
		else	{
			LoginPanelController	*aController	;
			
				aController = [[LoginPanelController alloc] init];
				[aController setConnection:self];
								
				[NSApp beginSheet:[aController window]
				modalForWindow:hostWindow
				modalDelegate:nil
				didEndSelector:nil
				contextInfo:nil];
				
				[NSApp runModalForWindow:[aController window]];
				
				[NSApp endSheet:[aController window]];
				[[aController window] orderOut:self];
				
				*returnCode = [aController errorCode]	;
				[aController close];	
				return [aController connectionOK]	;
		}
}

- (BOOL)connectAskingUserWithDefaults:(NSDictionary *)defaults return:(int *)returnCode forWindow:(NSWindow *)hostWindow
{
		if (hostWindow == nil)	{
			LoginWindowController	*aController	;
			
				aController = [[LoginWindowController alloc] initWithDefaults:defaults];
				[aController showWindow:self];
				[aController setConnection:self];
				
				[NSApp runModalForWindow:[aController window]]	;
				
				*returnCode = [aController errorCode]	;
				[aController close];	
				return [aController connectionOK]	;
		}
		else	{
			LoginPanelController	*aController	;
			
				aController = [[LoginPanelController alloc] initWithDefaults:defaults];
				[aController setConnection:self];
								
				[NSApp beginSheet:[aController window]
				modalForWindow:hostWindow
				modalDelegate:nil
				didEndSelector:nil
				contextInfo:nil];
				
				[NSApp runModalForWindow:[aController window]];
				
				[NSApp endSheet:[aController window]];
				[[aController window] orderOut:self];
				
				*returnCode = [aController errorCode]	;
				[aController close];	
				return [aController connectionOK]	;
		}
}

-(PGconn *)getPGconn;
{
#if	DEMO_MODE
		NSLog(@"WARNING PostgreSQL FrameWork: getPGconn is not implemented in DEMO version - returning nil !");
		return nil	;
#else
		return _conn	;
#endif
}

- (NSString *)postgreSQLVersion
{
	PGresult 	*result	;
	NSString	*res	;
	
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		result = PQexec(_conn,"SELECT version()");
		if (PQntuples(result) > 0)	{
			res = [NSString stringWithCString:PQgetvalue(result, 0, 0)]	;
			PQclear(result)	;
		}
		else
			res = [NSString stringWithCString:"UNKNOWN"]	;
			
		return res	;
}

- (NSString *)frameworkVersion
{
	NSDictionary	*aDict = [[NSBundle bundleForClass:[PostgreSQL class]] infoDictionary]	;
		
		return [[[aDict objectForKey:@"CFBundleExecutable"] stringByAppendingString:@" Framework "] stringByAppendingString:[aDict objectForKey:@"CFBundleVersion"]]	;
}

-(const char *)databaseName
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQdb(_conn)	;
}

-(const char *)hostName
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQhost(_conn)	;
}

-(const char *)loginName
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQuser(_conn)	;
}

-(const char *)password
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQpass(_conn)	;
}

-(const char *)port
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQport(_conn)	;
}

-(const char *)tty
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQtty(_conn)	;
}

-(const char *)options
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQoptions(_conn)	;
}

-(int)socket
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQsocket(_conn)	;
}

-(int)backendPID
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQbackendPID(_conn)	;
}

-(int)clientEncoding
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQclientEncoding(_conn)	;
}

-(int)setClientEncoding:(const char *)encoding
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(encoding != nil,nilArgumentMsg)	;
		}
		return PQsetClientEncoding(_conn,encoding)	;
}

-(void)_bindAny:(int)col address:(void *)itemAddr type:(TBindItemType)itemType  bindingArray:(NSMutableArray *)bindingArray
{
		if (_debugMode)	{
			NSAssert(bindingArray != nil,nilBindingArray)	;
		}
		
		if (col >= [bindingArray count])	{
			int	i ;
			
			for(i=[bindingArray count];i<=col;i++)
				[bindingArray addObject:[PgSQLBindItem newWithAddress:nil type:bindItem_Unknown]]	;
		}
		
		if (col > 0)
			[bindingArray replaceObjectAtIndex:(unsigned)col withObject:[PgSQLBindItem newWithAddress:itemAddr type:itemType]]	;
		else
			[bindingArray addObject:[PgSQLBindItem newWithAddress:itemAddr type:itemType]]	;
}

-(void)bindDouble:(double *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Double bindingArray:_bindingArray]	;
}

-(void)bindDouble:(double *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Double bindingArray:_bindingArray]	;
}

-(void)bindFloat:(float *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Float bindingArray:_bindingArray]	;
}

-(void)bindFloat:(float *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Float bindingArray:_bindingArray]	;
}

-(void)bindShort:(short *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Short bindingArray:_bindingArray]	;
}

-(void)bindShort:(short *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Short bindingArray:_bindingArray]	;
}

-(void)bindInteger:(int *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Integer bindingArray:_bindingArray]	;
}

-(void)bindInteger:(int *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Integer bindingArray:_bindingArray]	;
}

-(void)bindLong:(long *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Long bindingArray:_bindingArray]	;
}

-(void)bindLong:(long *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Long bindingArray:_bindingArray]	;
}

-(void)bindLongLong:(long long *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_LongLong bindingArray:_bindingArray]	;
}

-(void)bindLongLong:(long long *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_LongLong bindingArray:_bindingArray]	;
}

-(void)bindString:(char *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_CString bindingArray:_bindingArray]	;
}

-(void)bindString:(char *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_CString bindingArray:_bindingArray]	;
}

-(void)bindBinaryString:(BSTRING *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_BString bindingArray:_bindingArray]	;
}

-(void)bindBinaryString:(BSTRING *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_BString bindingArray:_bindingArray]	;
}

-(void)bindText:(TEXT *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Text bindingArray:_bindingArray]	;
}

-(void)bindText:(TEXT *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Text bindingArray:_bindingArray]	;
}

-(void)bindBoolean:(BOOL *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_BOOL bindingArray:_bindingArray]	;
}

-(void)bindBoolean:(BOOL *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_BOOL bindingArray:_bindingArray]	;
}

-(void)bindChar:(char *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Char bindingArray:_bindingArray]	;
}

-(void)bindChar:(char *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Char bindingArray:_bindingArray]	;
}

-(void)bindBinary:(Oid *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Oid bindingArray:_bindingArray]	;
}

-(void)bindBinary:(Oid *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Oid bindingArray:_bindingArray]	;
}

- (void)bindPolygon:(POLYGON **)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Polygon bindingArray:_bindingArray]	;
}

- (void)bindPolygon:(POLYGON **)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_Polygon bindingArray:_bindingArray]	;
}

- (void)bindLSeg:(LSEG *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_LSeg bindingArray:_bindingArray]	;
}

- (void)bindLSeg:(LSEG *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_LSeg bindingArray:_bindingArray]	;
}

- (void)bindPath:(PATH **)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Path bindingArray:_bindingArray]	;
}

- (void)bindPath:(PATH **)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_Path bindingArray:_bindingArray]	;
}

- (void)bindLine:(LINE *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Line bindingArray:_bindingArray]	;
}

- (void)bindLine:(LINE *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_Line bindingArray:_bindingArray]	;
}

- (void)bindBox:(BOX *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Box bindingArray:_bindingArray]	;
}

- (void)bindBox:(BOX *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_Box bindingArray:_bindingArray]	;
}

- (void)bindCircle:(CIRCLE *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Circle bindingArray:_bindingArray]	;
}

- (void)bindCircle:(CIRCLE *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_Circle bindingArray:_bindingArray]	;
}

- (void)bindPoint:(POINT *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Point bindingArray:_bindingArray]	;
}

- (void)bindPoint:(POINT *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_Point bindingArray:_bindingArray]	;
}

- (void)bindCIdr:(unsigned char *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_Cidr bindingArray:_bindingArray]	;
}

- (void)bindCIdr:(unsigned char *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_Cidr bindingArray:_bindingArray]	;
}

- (void)bindInet:(unsigned char *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_INet bindingArray:_bindingArray]	;
}

- (void)bindInet:(unsigned char *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_INet bindingArray:_bindingArray]	;
}

- (void)bindMACaddr:(unsigned char *)var;
{
		[self _bindAny:-1 address:(void *)var type:bindItem_MACAddr bindingArray:_bindingArray]	;
}

- (void)bindMACaddr:(unsigned char *)var column:(int)col;
{
		[self _bindAny:col address:(void *)var type:bindItem_MACAddr bindingArray:_bindingArray]	;
}

- (void)bindBIT:(BOOLARRAY *)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_BIT bindingArray:_bindingArray]	;
}

- (void)bindBIT:(BOOLARRAY *)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_BIT bindingArray:_bindingArray]	;
}

- (void)bindNSDecimal:(NSDecimalNumber **)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_NSDecimal bindingArray:_bindingArray]	;
}

- (void)bindNSDecimal:(NSDecimalNumber **)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_NSDecimal bindingArray:_bindingArray]	;
}

- (void)bindNSNumber:(NSNumber **)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_NSNumber bindingArray:_bindingArray]	;
}

- (void)bindNSNumber:(NSNumber **)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_NSNumber bindingArray:_bindingArray]	;
}

- (void)bindNSString:(NSString **)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_NSString bindingArray:_bindingArray]	;
}

- (void)bindNSString:(NSString **)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_NSString bindingArray:_bindingArray]	;
}

- (void)bindNSData:(NSData **)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_NSData bindingArray:_bindingArray]	;
}

- (void)bindNSData:(NSData **)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_NSData bindingArray:_bindingArray]	;
}

- (void)bindNSArray:(NSArray **)var
{
		[self _bindAny:-1 address:(void *)var type:bindItem_NSArray bindingArray:_bindingArray]	;
}

- (void)bindNSArray:(NSArray **)var column:(int)col
{
		[self _bindAny:col address:(void *)var type:bindItem_NSArray bindingArray:_bindingArray]	;
}

-(NSString *)escapeString:(const char *)inSource
{
	int	l = strlen(inSource)	;
	char	*buffer = malloc( 2*l + 1)	;
	NSString	*result	;
	
		if (buffer == nil)
			return nil	;
			
		PQescapeString(buffer,inSource,l)	;
		result = [NSString stringWithCString:buffer]	;
		free(buffer)	;
		return result	;
}

-(NSData *)escapeBinary:(const unsigned char *)inSource length:(size_t)len
{
	size_t	l		;
	unsigned char *buffer = PQescapeBytea((unsigned char *)inSource,len,&l);
	NSData	*result	;
	
		result = [NSData dataWithBytes:buffer length:l]	;
		free(buffer)	;
		return result	;
}

#if	SUPPORT_73
-(NSData *)unescapeBinary:(const unsigned char *)inSource length:(size_t)len
{
	size_t	l		;
	unsigned char *buffer = PQunescapeBytea((unsigned char *)inSource,len,&l);
	NSData	*result	;
	
		result = [NSData dataWithBytes:buffer length:l]	;
		free(buffer)	;
		return result	;
}
#endif

-(const char *)serverMessage
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		return PQerrorMessage(_conn)	;
}

- (const char *)connectErrorMessage:(int)errorCode
{
		return PQresStatus(errorCode)	;
}

-(BOOL)beginTransaction
{
#if	DEMO_MODE
		NSLog(@"WARNING PostgreSQL FrameWork: beginTransaction is NOT implemented in DEMO version !")	;
		return	LASTRESULT_ISOK;
#else
	BOOL	result	;
	
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
		++_transactionLevel	;
		_lastResult = PQexec(_conn,"BEGIN");
		result = LASTRESULT_ISOK;
		CLEAR_LASTRESULT	;
		return result	;
#endif
}

-(BOOL)commitTransaction
{
#if	DEMO_MODE
		NSLog(@"WARNING PostgreSQL FrameWork: commitTransaction is NOT implemented in DEMO version !")	;
		return	LASTRESULT_ISOK;
#else	
	BOOL	result	;

		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert([self transactionInProgress],noTransactionMsg)	;
		}
		--_transactionLevel	;
		_lastResult = PQexec(_conn,"END");
		result = LASTRESULT_ISOK;
		CLEAR_LASTRESULT	;
		return result	;
#endif
}

-(BOOL)rollbackTransaction
{
#if	DEMO_MODE
		NSLog(@"WARNING PostgreSQL FrameWork: rollbackTransaction is NOT implemented in DEMO version !")	;
		return	LASTRESULT_ISOK;
#else
	BOOL	result	;
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert([self transactionInProgress],noTransactionMsg)	;
		}
		--_transactionLevel	;
		_lastResult = PQexec(_conn,"ROLLBACK");
		result = LASTRESULT_ISOK;
		CLEAR_LASTRESULT	;
		return result	;
#endif
}

- (BOOL)transactionInProgress
{
		return _transactionLevel > 0	;
}

-(PgSQLResult *)getResultSet
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}
	
	PGresult	*res = _lastResult	;

		_lastResult = nil	;
		_curRowIndex = -1 	; 
		_rowsInResult = 0	; 
		_fieldsInResult = 0	;
			
	PgSQLResult	*pgResult = [PgSQLResult newWithPgResult:res forConnection:self]	;
	
		[_results addObject:pgResult]	;
		
		return pgResult	;
}

-(int)rowsAffected
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}
		return PQntuples(_lastResult);
}

-(int)rowsAffectedByCommand
{
	int	result = 0	;

		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}
	char *s = PQcmdTuples(_lastResult);
	
		if (s && *s)
			sscanf(s,"%d",&result)	;
			
		return result	;
}

-(void)clearCommands
{
		[_bindingArray release]	;
		_bindingArray = nil		;
		[_cmdBuffer release]		;
		_cmdBuffer = [[NSMutableString string] retain]	;
		_commandExecuted = NO	;
}

-(const char *)commandBuffer
{
		return [_cmdBuffer cString]	;
}

-(BOOL)executeCommand
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
			
		if (_asyncCommandInProgress)	{
			NSLog(@"executeAsyncCommand in progress")	;
			return NO	;
		}
		
		CLEAR_LASTRESULT;
		_asyncCommandResultAvailable = NO	;
		_lastResult = PQexec(_conn,[self commandBuffer])	;
		_rowsInResult = PQntuples(_lastResult)	;
		_fieldsInResult = PQnfields(_lastResult)	;
		if (_bindingArray != nil)
			[_bindingArray release]	;
		_bindingArray = [[NSMutableArray array] retain]	;			
		_commandExecuted = YES	;
		return LASTRESULT_ISOK;
}

-(BOOL)executeCommandWithCursor:(const char *)cursorName binary:(BOOL)binary
{
#if	!DEMO_MODE
	char	command[512]	;
	
		if (_debugMode)
			NSAssert([self transactionInProgress],noTransactionMsg)	;
			
		sprintf(command,"DECLARE %s%s CURSOR FOR ",cursorName,(binary)? (" BINARY"):(""));
		[_cmdBuffer insertString:[NSString stringWithCString:command] atIndex:0]	;
#endif
		return [self executeCommand]	;		
}

-(BOOL)executeAsyncCommand
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;

		if (_asyncThread == nil)	{
			NSLog(@"executeAsyncCommand no _asyncThread")	;
			return NO	;
		}
	
		if (_asyncCommandInProgress)	{
			NSLog(@"executeAsyncCommand in progress")	;
			return NO	;
		}
		CLEAR_LASTRESULT;
		_asyncCommandResultAvailable = NO	;
		if (_bindingArray != nil)
			[_bindingArray release]	;
		_bindingArray = [[NSMutableArray array] retain]	;			
		_commandExecuted = YES	;
		[_asyncThread executeAsyncCommand:_cmdBuffer]	;
		_asyncCommandInProgress = YES	;
		
		return _asyncCommandInProgress	;
}

-(BOOL)executeAsyncCommandWithCursor:(const char *)cursorName binary:(BOOL)binary
{
#if	!DEMO_MODE
	char	command[512]	;
	
		if (_debugMode)
			NSAssert([self transactionInProgress],noTransactionMsg)	;
			
		sprintf(command,"DECLARE %s%s CURSOR FOR ",cursorName,(binary)? (" BINARY"):(""));
		[_cmdBuffer insertString:[NSString stringWithCString:command] atIndex:0]	;
#endif
		return [self executeAsyncCommand]	;		
}

-(BOOL)cancelRequest	
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
	
		if (_asyncCommandInProgress)	{
			return (1 == PQrequestCancel(_conn))	;
		}
		
		if (_debugMode)
			NSLog(@"WARNING PostgreSQL FrameWork: cancelreuqest called when no async command in progress!")	;
			
		return YES	;
}

-(BOOL)closeCursor:(const char *)cursorName
{
#if	DEMO_MODE
		return YES	;
#else
	char	command[512]	;
	PGresult 	*result	;
	BOOL		res	;
	
		if (_debugMode)
			NSAssert([self transactionInProgress],noTransactionMsg)	;
			
		sprintf(command,"CLOSE %s",cursorName);
		result = PQexec(_conn,command)	;
		res = PQresultStatus(result) == PGRES_COMMAND_OK	;
		PQclear(result)	;
		return res	;
#endif
}

-(BOOL)bufferHasCommands
{
		return (_asyncCommandInProgress == NO) && (_commandExecuted == NO) && (strlen([self commandBuffer]) > 0)	;
}

-(void)makeCommand:(const char *)cmd
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(cmd != nil, nilArgumentMsg)	;
		}
		[_cmdBuffer appendString:[NSString stringWithCString:cmd]]	;
}

-(void)makeCommandWithString:(NSString *)cmd
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(cmd != nil, nilArgumentMsg)	;
		}
		[_cmdBuffer appendString:cmd]	;
}

-(void)makeCommandf:(const char *)format, ...
{
	char	command[1024]	;
	va_list args;                            

		va_start( args, format );                
		vsprintf(command, format, args)	;
		va_end(args)	;
		[self makeCommand:command]	;
}

-(void)vmakeCommandf:(const char *)format args:(va_list)args
{
	char	command[1024]	;

		vsprintf(command, format, args)	;
		[self makeCommand:command]	;
}

-(id)_fetchFieldValue:(PGresult *)result atIndex:(int)rowIndex forField:(int)fieldIndex binary:(BOOL)binary
{
		
		if (binary)
			return [NSData dataWithBytes:PQgetvalue(result, rowIndex, fieldIndex) length:PQgetlength(result, rowIndex, fieldIndex)]	;

	int	type = PQftype(result, fieldIndex)	;
	int	len = PQgetlength(result,rowIndex, fieldIndex)	;
	id	res	;
	
		switch(type)	{
			case	INT8OID :
			case	INT4OID :
			case	INT2OID :
			case	FLOAT4OID :
			case	NUMERICOID :
			case	FLOAT8OID :
			case	CHAROID :
			case	BOOLOID :
				return [self _bindAnyNumber:result atIndex:rowIndex type:type forField:fieldIndex binary:0]	;
				break	;
				
			case	TEXTOID	:	{
					TEXT	txt	;
					
					[self _bindText:result atIndex:rowIndex forField:fieldIndex binary:binary addr:&txt]	;
					
					if (txt.text != NULL)	{
						res = [NSString stringWithCString:txt.text]	;
						free(txt.text)	;
					}
					else
						res = [NSString string]	;
						
					return res	;
				}
				break	;
				
			case	VARCHAROID	:	
			case	BPCHAROID	:	{
				char	*s = malloc(len + 1)	;
				
					if (s != NULL)	{
						[self _bindString:result atIndex:rowIndex forField:fieldIndex binary:binary addr:s]	;
						res = [NSString stringWithCString:s]	;
						free(s)	;
					}		
					else
						res = [NSString string]	;
						
					return res	;
				}
				break	;

			case	BYTEAOID	:	{
					BSTRING	bstring	;
					
					[self _bindBinaryString:result atIndex:rowIndex forField:fieldIndex binary:binary addr:&bstring]	;
					if (bstring.bytes)	{
						res = [NSMutableData dataWithBytes:bstring.bytes length:bstring.size]	;

						free(bstring.bytes)	;
					}
					else
						res = [NSData data]	;
						
					return res	;
				}
				break;

			case	OIDOID	:	{
					unsigned int	ui	;
	
					[self _bindInteger:result atIndex:rowIndex forField:fieldIndex binary:binary addr:(int *)&ui]	;
				
					return [self retrieveBinary:ui]	;
				}
				break;
		}
		
		return [NSString stringWithCString: PQgetvalue(result, rowIndex, fieldIndex) ];
}

-(BOOL)isColumnNULL:(int)fieldIndex
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

		if ((_curRowIndex < 0) || (_curRowIndex >= _rowsInResult))	{
			if (_debugMode)
				NSLog(@"PostgreSQL FrameWork : isColumnNULL called with invalid _curRowIndex !");
			return YES	;
		}
		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if (_debugMode)
				NSLog(@"PostgreSQL FrameWork : isColumnNULL called with invalid fieldIndex !");
			return YES	;
		} 
		return PQgetisnull(_lastResult, _curRowIndex, fieldIndex)	;
}

-(int)_fetchNFields
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

		return _fieldsInResult	;
}

-(const char *)_fetchFieldName:(int)fieldIndex
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if (_debugMode)
				NSLog(@"PostgreSQL FrameWork : fetchFieldName called with invalid fieldIndex !");
			return "ERROR"	;
		} 
		return PQfname(_lastResult, fieldIndex)	;
}

-(const char *)_fetchFieldTypeName:(int)fieldIndex
{
	char		command[512], *res = "ERROR"	;
	PGresult 	*result	;
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if (_debugMode)
				NSLog(@"PostgreSQL FrameWork : fetchFieldType called with invalid fieldIndex !");
			return res	;
		} 

		sprintf(command,"SELECT typname FROM pg_type WHERE oid=%d",PQftype(_lastResult, fieldIndex))	;
		result = PQexec(_conn,command)	;
		if (PQntuples(result) > 0)	{
			res = PQgetvalue(result, 0, 0)	;
			PQclear(result)	;
		}
		else
			res = "UNKNOWN"	;
			
		return res	;
}

-(Oid)_fetchFieldType:(int)fieldIndex
{
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if (_debugMode)
				NSLog(@"PostgreSQL FrameWork : fetchFieldType called with invalid fieldIndex !");
			return 0	;
		} 
			
		return PQftype(_lastResult, fieldIndex)	;
}

static	char *bcd2str(const char *inBCD,char *outStr,int nDigits,int intPart)	{
	const char	*s	;
	char	*d	;
	int	i	;
	
		s = inBCD	;
		d = outStr	;
		
		for(i=0;i<nDigits;i++,s++)	{
			if (i == intPart)
				*d++ = '.'	;
			*d++ = ((*s & 0xF0) >> 4) + '0'	;
			++i	;
			if (i<nDigits)	{
				if (i == intPart)
					*d++ = '.'	;
				*d++ = (*s & 0x0F) + '0'	;
			}
		}
		
			
		*d = '\0'	;
		
		return outStr	;
}

-(void)_bindDouble:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(double *)addr
{
		if (binary)	{
			double	*p = (double *)PQgetvalue(result, rowIndex, field)	;
			if (p)
				*(double *)addr = *p	;
			else
				*(double *)addr = NAN	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			if (p && *p)
				sscanf(p,"%lf",(double *)addr)	;
			else
				*(double *)addr = NAN	;
		}
}

-(void)_bindFloat:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(float *)addr
{
		if (binary)	{
			float	*p = (float *)PQgetvalue(result, rowIndex, field)	;
			if (p)
				*(float *)addr = *p	;
			else
				*(float *)addr = NAN	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			if (p && *p)
				sscanf(p,"%f",(float *)addr)	;
			else
				*(float *)addr = NAN	;
		}
}

-(void)_bindDecimal2Double:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(double *)addr
{
		if (binary)	{
			unsigned char *p = (unsigned char *)PQgetvalue(result, rowIndex, field)	;
			if (p)	{
				char 	tempBuffer[256]	;
				short	weight = ((short *)p)[0]	;
				short	dscale =  ((short *)p)[2] & 0x00FF	;
				short	negative =  (((short *)p)[2] & 0xFF00) == 0x4000	;
				int		ndigits = weight + dscale + 1	;
				double	t	;
				
				p += 3 * sizeof(short)	;
				bcd2str(p,tempBuffer,ndigits,ndigits-dscale);
				
				sscanf(tempBuffer,"%lf",&t)	;
				if (negative)
					*(double *)addr = -t	;
				else
					*(double *)addr = t	;
			}
			else
				*(double *)addr = NAN	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			if (p && *p)
				sscanf(p,"%lf",(double *)addr)	;
			else
				*(double *)addr = NAN	;
		}
}
		
-(void)_bindNSDecimal:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(NSDecimalNumber **)addr
{
		*addr = nil	;
		
		if (binary)	{
			unsigned char *p = (unsigned char *)PQgetvalue(result, rowIndex, field)	;
			if (p)	{
				char 	tempBuffer[256], *s = tempBuffer	;
				short	weight = ((short *)p)[0]	;
				short	dscale =  ((short *)p)[2] & 0x00FF	;
				short	negative =  (((short *)p)[2] & 0xFF00) == 0x4000	;
				int		ndigits = weight + dscale + 1	;
				
				p += 3 * sizeof(short)	;
				if (negative)
					*s++ = '-'	;
				bcd2str(p,s,ndigits,ndigits-dscale);
				*(NSDecimalNumber **)addr = [NSDecimalNumber decimalNumberWithString:[NSString stringWithCString:tempBuffer] locale:_localUSDict]	;
			}
			else
				*(NSDecimalNumber **)addr = [NSDecimalNumber  notANumber]	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			*(NSDecimalNumber **)addr = [NSDecimalNumber decimalNumberWithString:[NSString stringWithCString:p] locale:_localUSDict]	;
		}
}

-(void)_bindBoolean:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(BOOL *)addr
{
		if (binary)	{
			BOOL *p = (BOOL *)PQgetvalue(result, rowIndex, field);
			
			if (p)
				*(BOOL *)addr = *p	;
			else
				*(BOOL *)addr = NO	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			char	aChar	;
			
			if (p && *p)	{
				sscanf(p,"%c",&aChar)	;
				*(BOOL *)addr = (aChar == 't') || (aChar == 'T') || (aChar == '1') || (aChar == 'y') || (aChar == 'Y')	;
			}
			else
				*(BOOL *)addr = NO	;
		}
}

-(void)_bindChar:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(char *)addr
{
		if (binary)	{
			char *p = (char *)PQgetvalue(result, rowIndex, field);
			
			if (p)
				*(char *)addr = *p	;
			else
				*(char *)addr = '\0'	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;

			if (p && *p)	
				sscanf(p,"%c",(char *)addr)	;
			else
				*(char *)addr = '\0'	;
		}
}

-(void)_bindShort:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(short *)addr
{
		if (binary)	{
			short *p = (short *)PQgetvalue(result, rowIndex, field);
			
			if (p)
				*(short *)addr = *p	;
			else
				*(short *)addr = 0	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			if (p && *p)	{
				int	t	;
				
				sscanf(p,"%d",&t)	;
				*(short *)addr = (short)t	;
			}
			else
				*(short *)addr = 0	;
		}
}

-(void)_bindInteger:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(int *)addr
{
		if (binary)	{
			int *p = (int *)PQgetvalue(result, rowIndex, field);
			
			if (p)
				*(int *)addr = *p	;
			else
				*(int *)addr = 0	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			if (p && *p)
				sscanf(p,"%d",(int *)addr)	;
			else
				*(int *)addr = 0	;
		}
}

-(void)_bindLongLong:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(long long *)addr
{
		if (binary)	{
			long long *p = (long long *)PQgetvalue(result, rowIndex, field);
			
			if (p)
				*(long long *)addr = *p	;
			else
				*(long long *)addr = 0	;
		}
		else	{
			char	*p = PQgetvalue(result, rowIndex, field)	;
			if (p && *p)
				sscanf(p,"%lld",(long long *)addr)	;
			else
				*(long long *)addr = 0	;
		}
}

-(void)_bindText:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(TEXT *)addr
{
	char *p = (char *)PQgetvalue(result, rowIndex, field);

		addr->size = 0	;
		addr->text = NULL	;

		if (p == NULL)	{
			return	;
		}
		
		if (binary)	{
			int		len = PQgetlength(result, rowIndex, field);
			char	*pval = malloc(len+1)	;
			if (pval != NULL)	{
				strncpy(pval,p,len)	;
				pval[len] = '\0'	;
			}
			addr->size = len	;
			addr->text = pval	;
		}
		else	if (*p)	{
			int		len = PQgetlength(result, rowIndex, field);
			char	*pval = malloc(len+1)	;

			if (pval != NULL)
				strcpy(pval,p)	;
			
			addr->size = len	;
			addr->text = pval	;
		}
}

-(void)_bindString:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(char *)addr
{
	char *p = (char *)PQgetvalue(result, rowIndex, field);
	
		*addr = '\0'	;
		if (p)
			if (binary)	{
				strncpy((char *)addr,p,PQgetlength(_lastResult,_curRowIndex, field))	;
			}
			else	{
				strcpy((char *)addr,p)	;
			}
}

-(void)_bindBinaryString:(PGresult *)result atIndex:(int)rowIndex forField:(int)field binary:(int)binary addr:(BSTRING *)addr
{
	unsigned char *p = (unsigned char *)PQgetvalue(result, rowIndex, field);

		addr->size = 0	;
		addr->bytes = NULL	;

		if (p)
			if (binary)	{
				int		len = PQgetlength(result, rowIndex, field);
				unsigned char	*pval = (unsigned char *)malloc(len)	;
			
					if (pval != NULL)	{
						memmove(pval,p,len)	;
					}
					addr->size = len	;
					addr->bytes = pval	;
			}
			else	{
				if (!PQgetisnull(result, rowIndex, field))	{
					int		len = PQgetlength(result, rowIndex, field);
					unsigned char	*pval = (unsigned char *)malloc(len)	;
	
					if (pval != NULL)	{
						unsigned char *pval1 = pval, *p1 = p	;
						int i	;
						
						for(i=0 ; i<len ;i++)	{
							if (*p1 == '\\')	{
								*pval1 = (p1[1]-'0') * 64 + (p1[2]-'0') * 8 + (p1[3]-'0')	;
								p1 += 3	;
								i += 3	;
							}
							else
								*pval1 = *p1	;
							++pval1	;
							++p1	;
						}
						if (len != pval1 - pval)	{
							len = pval1 - pval	;
							pval = realloc(pval,len)	;
						}
					
						addr->size = len	;
						addr->bytes = pval	;
						
					}
				}
			}
}

-(id)_bindAnyNumber:(PGresult *)result atIndex:(int)rowIndex type:(int)type forField:(int)field binary:(int)binary
{
	id	res = nil	;
	
		switch(type)	{
			case	BOOLOID	:	{
				BOOL	b	;
				
					[self _bindBoolean:result atIndex:rowIndex forField:field binary:binary addr:&b]	;
					res = [NSNumber numberWithBool:b]	;
				}
				break	;
				
			case	CHAROID	:	{
				char	c	;
				
					[self _bindChar:result atIndex:rowIndex forField:field binary:binary addr:&c]	;
					res = [NSNumber numberWithChar:c]	;
				}
				break	;
				
			case	INT8OID	:	{
				long long	ll	;
				
					[self _bindLongLong:result atIndex:rowIndex forField:field binary:binary addr:&ll]	;
					res = [NSNumber numberWithLongLong:ll]	;
				}
				break	;
				
			case	INT4OID	:	{
				int	i	;
				
					[self _bindInteger:result atIndex:rowIndex forField:field binary:binary addr:&i]	;
					res = [NSNumber numberWithInt:i]	;
				}

				break	;
				
			case	INT2OID	:	{
				short	i	;
				
					[self _bindShort:result atIndex:rowIndex forField:field binary:binary addr:&i]	;
					res = [NSNumber numberWithInt:(int)i]	;
				}

				break	;
				
			case	OIDOID	:	{
				unsigned int	ui	;
				
					[self _bindInteger:result atIndex:rowIndex forField:field binary:binary addr:(int *)&ui]	;
					res = [NSNumber numberWithUnsignedInt:ui]	;
				}

				break	;
			
			case	FLOAT4OID	:	{
				float	f	;
				
					[self _bindFloat:result atIndex:rowIndex forField:field binary:binary addr:&f]	;
					res = [NSNumber numberWithFloat:f]	;
				}
				break	;
			
			case	FLOAT8OID	:	{
				double d	;
				
					[self _bindDouble:result atIndex:rowIndex forField:field binary:binary addr:&d]	;
					res = [NSNumber numberWithDouble:d]	;
				}
				break	;
				
			case	NUMERICOID	:
				[self _bindNSDecimal:result atIndex:rowIndex forField:field binary:binary addr:&res]	;
				break	;
		}
		
		return res	;
}

static void	dumpHexa(unsigned char *bytes, int len)
{
	int	i	;
	unsigned char *p = bytes	;
	
		for(i=0;i<len;i++,p++)	{
			printf("%X ",*p)	;
		}
		printf("\n")	;
}

static void fillLevel(NSMutableArray *inArray, int curLevel,int *dimensions,int nDims,Oid type,void **dataHdl)
{
	int		i	;
	NSMutableArray	*newArray	;
	
		if (curLevel == nDims-1)	{	//last level
			switch(type)	{
				case	BOOL_ARRAY_OID	:	{
						unsigned char	*p = *(unsigned char **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++,p++)
							[inArray addObject:[NSNumber numberWithBool:(BOOL)*p]]	;
						*dataHdl = (void *)p	;
					}
					break	;
				
				case	CHAR_ARRAY_OID	:	{
						char	*p = *(char **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++,p++)
							[inArray addObject:[NSNumber numberWithChar:*p]]	;
						*dataHdl = (void *)p	;
					}
					break	;
				
				case	SHORT_ARRAY_OID	:	{
						short	*p = *(short **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++,p++)
							[inArray addObject:[NSNumber numberWithInt:(int)*p]]	;
						*dataHdl = (void *)p	;
					}
					break	;
				
				case	INT_ARRAY_OID	:
				case	OID_ARRAY_OID	:	{
						int	*p = *(int **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++,p++)
							[inArray addObject:[NSNumber numberWithInt:*p]]	;
						*dataHdl = (void *)p	;
					}
					break	;
				
				case	LONGLONG_ARRAY_OID	:	{
						long long	*p = *(long long **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++,p++)
							[inArray addObject:[NSNumber numberWithLongLong:*p]]	;
						*dataHdl = (void *)p	;
					}
					break	;
				
				case	FLOAT_ARRAY_OID	:	{
						float	*p = *(float **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++,p++)
							[inArray addObject:[NSNumber numberWithFloat:*p]]	;
						*dataHdl = (void *)p	;
					}
					break	;
				
				case	DOUBLE_ARRAY_OID	:	{
						double	*p = *(double **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++,p++)
							[inArray addObject:[NSNumber numberWithDouble:*p]]	;
						*dataHdl = (void *)p	;
					}
					break	;
					
				case	STRING_ARRAY_OID	:
				case	VSTRING_ARRAY_OID	:	{
						char	*p = *(char **)dataHdl	;
						
						for(i=0;i<dimensions[curLevel];i++)	{
							int	len = *(int *)p	;
							
							p += sizeof(int)	;
							[inArray addObject:[NSString stringWithCString:p]]	;
							 
							if (len % 4)
								len = ((len / 4) + 1) * 4	;
								
							p += (len - sizeof(int))	;
							
						}
						*dataHdl = (void *)p	;
					}
					break	;
			}
		}
		else	{
			for(i=0;i<dimensions[curLevel];i++)	{
				newArray = [NSMutableArray arrayWithCapacity:dimensions[curLevel+1]]	;
				[inArray addObject:newArray]	;
				fillLevel(newArray,curLevel+1,dimensions,nDims,type,dataHdl)	;
			}
		}
}

static void fillArray(NSMutableArray *inArray, int curLevel, int *dimensions, int nDims, int type, char **inString)
{
	int		i	;
	NSMutableArray	*newArray	;
	char	*s = *inString	;
	char	*s0	;
	
		while ((*s == '{') || (*s == '}') || (*s == ',')) ++s;
//	printf("fillarray %d %s after skipping %s\n",curLevel,*inString,s)	;
		
		if (curLevel == nDims-1)	{	// last level
			switch(type)	{
				case	BOOL_ARRAY_OID	:	{
						char	c	;
						
						for(i=0;i<dimensions[nDims-1];i++)	{
							sscanf(s,"%c",&c)	;
							[inArray addObject:[NSNumber numberWithBool:(c == 't')]]	;
							s0 = strsep(&s,",")	;
						}
					}
					break	;
				
				case	CHAR_ARRAY_OID	:	{
						char	c	;
						
						for(i=0;i<dimensions[nDims-1];i++)	{
							sscanf(s,"%c",&c)	;
							[inArray addObject:[NSNumber numberWithChar:c]]	;
							s0 = strsep(&s,",")	;
						}
					}
					break	;
				
				case	INT_ARRAY_OID	:
				case	OID_ARRAY_OID	:	
				case	SHORT_ARRAY_OID	:	{
						int		c	;
						
						for(i=0;i<dimensions[nDims-1];i++)	{
							sscanf(s,"%d",&c)	;
							[inArray addObject:[NSNumber numberWithInt:c]]	;
							s0 = strsep(&s,",")	;
						}
					}
					break	;
								
				case	LONGLONG_ARRAY_OID	:	{
						long long		c	;
						
						for(i=0;i<dimensions[nDims-1];i++)	{
							sscanf(s,"%lld",&c)	;
							[inArray addObject:[NSNumber numberWithLongLong:c]]	;
							s0 = strsep(&s,",")	;
						}
					}
					break	;
				
				case	FLOAT_ARRAY_OID	:	{
						float		c	;
						
						for(i=0;i<dimensions[nDims-1];i++)	{
							sscanf(s,"%f",&c)	;
							[inArray addObject:[NSNumber numberWithFloat:c]]	;
							s0 = strsep(&s,",")	;
						}
					}
					break	;
				
				case	DOUBLE_ARRAY_OID	:	{
						double		c	;
						
						//	printf("DOUBLE_ARRAY_OID s %s\n",s	);
						for(i=0;i<dimensions[nDims-1];i++)	{
							sscanf(s,"%lf",&c)	;
							[inArray addObject:[NSNumber numberWithDouble:c]]	;
							s0 = strsep(&s,",")	;
							//	printf("DOUBLE_ARRAY_OID i %d s %s\n",i, s	);
						}
					}
					break	;
					
				case	STRING_ARRAY_OID	:
				case	VSTRING_ARRAY_OID	:	{
					char 	eos, *s1, *buffer, *d	;
					int		size = 1024, len	;
					
						buffer = malloc(1024)	;
						if (buffer == nil)
							break	;
							
						if (*s == '"')	{
							eos = '"'	;
							++s	;
						}
						else
							eos = ','	;
							
//printf("VSTRING_ARRAY_OID %d ",dimensions[nDims-1])	;
						for(i=0;i<dimensions[nDims-1];i++)	{
							s1 = s	;
							while (*s1 != eos)	{
								if (*s1 == '\\')
									++s1	;
								++s1	;
							}
							len = s1 - s	;
							if (len+1 > size)	{
								char *t = realloc(buffer,len+64)	;
								if (t == NULL)	{
									free(buffer)	;
									break	;
								}
								buffer = t	;
								size = len+64	;
							}
							d = buffer	;
							while (s != s1)	{
								if (*s == '\\')
									++s	;
								*d++ = *s++	;
							}
							*d = '\0'	;
							[inArray addObject:[NSString stringWithCString:buffer]]	;
//printf("buffer %s\n",buffer)	;
							s = s1 + 1	;
						}
						free(buffer)	;
					}
					break	;
			}
			*inString = s	;
		}
		else	{	// not last level
//	if (VSTRING_ARRAY_OID == type) printf("dimensions[curLevel] %d\n",dimensions[curLevel])	;
			for(i=0;i<dimensions[curLevel];i++)	{
				newArray = [NSMutableArray arrayWithCapacity:dimensions[curLevel+1]]	;
				[inArray addObject:newArray]	;
				fillArray(newArray,curLevel+1,dimensions,nDims,type,inString)	;
			}
		}
}

-(void)_fetchData
{
		[self _fetchData:_lastResult atIndex:_curRowIndex withBinding:_bindingArray]	;
}

-(void)_fetchData:(PGresult *)forResult atIndex:(int)rowIndex withBinding:(NSArray *)bindingArray
{
	int	field	;
	int	binary = PQbinaryTuples(forResult)	;
	int	type	;
	
		for(field=0;field<[bindingArray count];field++)	{
			PgSQLBindItem	*item = [bindingArray objectAtIndex:field]	;
			void	*addr = [item address]	;
			
				type = PQftype(forResult, field);
				switch([item type])	{
					case	bindItem_Unknown	:
						break	;
						
					case	bindItem_Double	:	{
						
							if (_debugMode)	{
								NSAssert((FLOAT8OID == type) || (NUMERICOID == type),bindTypeError)	;
							}
							
							if (type == FLOAT8OID)
								[self _bindDouble:forResult atIndex:rowIndex forField:field binary:binary addr:(double *)addr]	;
							else	{
								NSLog(@"WARNING binding double on NUMERIC data type may lose precision !")	;
								[self _bindDecimal2Double:forResult atIndex:rowIndex forField:field binary:binary addr:(double *)addr]	;
							}	
						}
						break	;

					case	bindItem_NSDecimal	:	{
							if (_debugMode)	{
								NSAssert(NUMERICOID ==  type,bindTypeError)	;
							}
							[self _bindNSDecimal:forResult atIndex:rowIndex forField:field binary:binary addr:(NSDecimalNumber **)addr];
						}
						break	;	
											
					case	bindItem_Float	:
						if (_debugMode)
							NSAssert(FLOAT4OID == type,bindTypeError)	;
							
						[self _bindFloat:forResult atIndex:rowIndex forField:field binary:binary addr:(float *)addr]	;
						break	;
						
					case	bindItem_Oid	:
						if (_debugMode)
							NSAssert(OIDOID == type,bindTypeError)	;
							
						[self _bindInteger:forResult atIndex:rowIndex forField:field binary:binary addr:(int *)addr]	;
						break	;

					case	bindItem_Short	:
						if (_debugMode)
							NSAssert(INT2OID == type,bindTypeError)	;
							
						[self _bindShort:forResult atIndex:rowIndex forField:field binary:binary addr:(short *)addr]	;
						break	;

					case	bindItem_Integer	:
						if (_debugMode)
							NSAssert(INT4OID == type,bindTypeError)	;
							
						[self _bindInteger:forResult atIndex:rowIndex forField:field binary:binary addr:(int *)addr]	;
						break	;

					case	bindItem_Long	:
						if (_debugMode)
							NSAssert(INT4OID == type,bindTypeError)	;
							
						[self _bindInteger:forResult atIndex:rowIndex forField:field binary:binary addr:(int *)addr]	;
						break	;

					case	bindItem_LongLong	:
						if (_debugMode)
							NSAssert(INT8OID == type,bindTypeError)	;
							
						[self _bindLongLong:forResult atIndex:rowIndex forField:field binary:binary addr:(long long *)addr]	;
						break	;

					case	bindItem_NSNumber	:	{
							
							if (_debugMode)	{
								NSAssert(
									(INT8OID == type) ||
									(INT4OID == type) ||
									(INT2OID == type) ||
									(OIDOID == type) ||
									(FLOAT4OID == type) ||
									(NUMERICOID == type) ||
									(FLOAT8OID == type) ||
									(CHAROID == type) ||
									(BOOLOID == type)
									,bindTypeError)	;
							}
							
							*(NSNumber **)addr = [self _bindAnyNumber:forResult atIndex:rowIndex type:type forField:field binary:binary]	;
							
						}
						break	;
						
					case	bindItem_CString	:
						if (_debugMode && binary)	{
							NSAssert((VARCHAROID == type) || (BPCHAROID == type),bindTypeError)	;
						}
							
						[self _bindString:forResult atIndex:rowIndex forField:field binary:binary addr:(char *)addr]	;
						break	;

					case	bindItem_Text	:
						if (_debugMode && binary)
							NSAssert(TEXTOID == type,bindTypeError)	;
							
						[self _bindText:forResult atIndex:rowIndex forField:field binary:binary addr:(TEXT *)addr]	;
						break	;

					case	bindItem_NSString	:	{
						
							*(NSString **)addr = nil	;
						
							if (type == TEXTOID)	{
								TEXT	txt	;
								
								[self _bindText:forResult atIndex:rowIndex forField:field binary:binary addr:&txt]	;
								
								if (txt.text != NULL)	{
									*(NSString **)addr = [NSString stringWithCString:txt.text]	;
									free(txt.text)	;
								}
							}
							else	if ((type == VARCHAROID) || (type == BPCHAROID))	{
								char	*s = malloc(PQgetlength(forResult,rowIndex, field) + 1)	;
								
									if (s != NULL)	{
										[self _bindString:forResult atIndex:rowIndex forField:field binary:binary addr:s]	;
										*(NSString **)addr = [NSString stringWithCString:s]	;
										free(s)	;
									}									
							}
							else	{
								id	result = [self _bindAnyNumber:forResult atIndex:rowIndex type:type forField:field binary:binary]	;
								
									if (result == nil)	{
										if (!binary)	{
											char	*p = PQgetvalue(forResult, rowIndex, field)	;
											
											if (p)
												*(NSString **)addr = [NSString stringWithCString:p]	;
										}
									}
									else	{
										*(NSString **)addr = [result descriptionWithLocale:_localUSDict]	; 
									}
							}
						}
						break	;
						
					case	bindItem_NSData	:	{
							*(NSData **)addr = nil	;
							
							if (type == BYTEAOID)	{
								BSTRING	bstring	;
								
								[self _bindBinaryString:forResult atIndex:rowIndex forField:field binary:binary addr:&bstring]	;
								if (bstring.bytes)	{
									*(NSData **)addr = [NSMutableData dataWithBytes:bstring.bytes length:bstring.size]	;

									free(bstring.bytes)	;
								}
							}
							else	if (type == OIDOID)	{
								unsigned int	ui	;
				
								[self _bindInteger:forResult atIndex:rowIndex forField:field binary:binary addr:(int *)&ui]	;
							
								*(NSData **)addr = [self retrieveBinary:ui]	;
							}
							else	{
								if (binary)	{
									unsigned char	*p = (unsigned char *)PQgetvalue(forResult, rowIndex, field)	;
										
									if (p)
										*(NSData **)addr = [NSMutableData dataWithBytes:p length:PQgetlength(forResult, rowIndex, field)]	;
								}
							}
						}
						break	;
						
					case	bindItem_BString	:
						if (_debugMode)
							NSAssert(BYTEAOID == type,bindTypeError)	;
							
						[self _bindBinaryString:forResult atIndex:rowIndex forField:field binary:binary addr:(BSTRING *)addr]	;
						break	;
					
					case	bindItem_BOOL	:
						if (_debugMode)
							NSAssert(BOOLOID == type,bindTypeError)	;
							
						[self _bindBoolean:forResult atIndex:rowIndex forField:field binary:binary addr:(BOOL *)addr]	;
						break	;
					
					case	bindItem_Char	:
						if (_debugMode)
							NSAssert(CHAROID == type,bindTypeError)	;
							
						[self _bindChar:forResult atIndex:rowIndex forField:field binary:binary addr:(char *)addr]	;
						break	;
					
					case	bindItem_Polygon	:
						if (_debugMode)
							NSAssert(POLYGONOID == type,bindTypeError)	;
							
						if (binary)	{
							int		plen = PQgetlength(forResult, rowIndex, field)	;
							char	*p = PQgetvalue(forResult, rowIndex, field);
							POLYGON	*pval = nil	;
							
								if (p && !PQgetisnull(forResult, rowIndex, field))	{
									pval = (POLYGON *)malloc(plen + VARHDRSZ)		;
							
									if (pval != nil)	{
										pval->size = plen	;
										memmove((char *)&pval->npts, p, plen)	;
									}
								}
								*(POLYGON **)addr = pval	;
						}
						else	{	// ((x,y),(x,y),...,(x,y))
							char	*p = PQgetvalue(forResult, rowIndex, field)	;
								
								if (p && *p)	{
									double	x, y	;
									char	*p1	;
									int	n = -1	;
									POLYGON	*pval		;
									size_t	size	;
									
									p1 = p	;
									while (*p1)	{
										if (*p1 == '(')
										 ++n	;
										++p1	;
									}
									
									size = sizeof(POLYGON) + (sizeof(POINT) * (n-1))	;
									pval = (POLYGON *)malloc(size)		;
									if (pval != NULL)	{
										double xmin, ymin, xmax, ymax	;
										
										xmin = ymin = HUGE_VAL	;
										xmax = ymax = -HUGE_VAL	;
										pval->npts = n	;

										n = 0	;
										p1 = strtok(p,",(")	;
										while (p1 != NULL)	{
											sscanf(p1,"%lf",&x)	;
											p1 = strtok(NULL,",(");
											sscanf(p1,"%lf",&y)	;
											pval->p[n].x = x	;
											pval->p[n].y = y	;
																						
											if (x > xmax)
												xmax = x	;
											if (x < xmin)
												xmin = x	;
											if (y > ymax)
												ymax = y	;
											if (y < ymin)
												ymin = y	;
											++n	;
											p1 = strtok(NULL,",(");
										}
										pval->size = size	;
										pval->boundbox.high.x = xmax	;
										pval->boundbox.high.y = ymax	;
										pval->boundbox.low.x = xmin	;
										pval->boundbox.low.y = ymin	;
									}
									
									*(POLYGON **)addr = pval	;
								}
								else
									*(POLYGON **)addr = NULL	;
						}
						break	;
					
					case	bindItem_LSeg	:
						if (_debugMode)
							NSAssert(LSEGOID == type,bindTypeError)	;
							
						if (binary)	{
							LSEG *p = (LSEG *)PQgetvalue(forResult, rowIndex, field);
							
							if (p && !PQgetisnull(forResult, rowIndex, field))
								*(LSEG *)addr = *p	;
							else	{
								((LSEG *)addr)->p[0].x = NAN	;
								((LSEG *)addr)->p[0].y = NAN	;
								((LSEG *)addr)->p[1].x = NAN	;
								((LSEG *)addr)->p[1].y = NAN	;
								((LSEG *)addr)->m = NAN	;
							}
						}
						else	{	// [(x1,y1),(x2,y2)]
							char	*p = PQgetvalue(forResult, rowIndex, field)	;

								if (p && *p)	{
									char	*p1	;
									double	t	;
									
										p1 = strtok(p,",([")	;
										sscanf(p1,"%lf",&t)	;
										((LSEG *)addr)->p[0].x = t	;
										p1 = strtok(NULL,",([")	;
										sscanf(p1,"%lf",&t)	;
										((LSEG *)addr)->p[0].y = t	;
										p1 = strtok(NULL,",([")	;
										sscanf(p1,"%lf",&t)	;
										((LSEG *)addr)->p[1].x = t	;
										p1 = strtok(NULL,",([")	;
										sscanf(p1,"%lf",&t)	;
										((LSEG *)addr)->p[1].y = t	;
										if (((LSEG *)addr)->p[0].x != ((LSEG *)addr)->p[1].x)
											((LSEG *)addr)->m = (((LSEG *)addr)->p[1].y - ((LSEG *)addr)->p[0].y) / (((LSEG *)addr)->p[1].x - ((LSEG *)addr)->p[0].x)	;
										else
											((LSEG *)addr)->m = HUGE_VAL	;
								}
								else	{
									((LSEG *)addr)->p[0].x = NAN	;
									((LSEG *)addr)->p[0].y = NAN	;
									((LSEG *)addr)->p[1].x = NAN	;
									((LSEG *)addr)->p[1].y = NAN	;
									((LSEG *)addr)->m = NAN	;
								}
						}
						break	;
						
					case bindItem_Box :
						if (_debugMode)
							NSAssert(BOXOID == type,bindTypeError)	;
							
						if (binary)	{
							BOX *p = (BOX *)PQgetvalue(forResult, rowIndex, field);
							
							if (p && !PQgetisnull(forResult, rowIndex, field))
								*(BOX *)addr = *p	;
							else	{
								((BOX *)addr)->high.x = NAN	;
								((BOX *)addr)->high.y = NAN	;
								((BOX *)addr)->low.x = NAN	;
								((BOX *)addr)->low.y = NAN	;
							}
						}
						else	{	// (x1,y1),(x2,y2)
							char	*p = PQgetvalue(forResult, rowIndex, field)	;

								if (p && *p)	{
									char	*p1	;
									double	t	;
									
										p1 = strtok(p,",(")	;
										sscanf(p1,"%lf",&t)	;
										((BOX *)addr)->high.x = t	;
										p1 = strtok(NULL,",(")	;
										sscanf(p1,"%lf",&t)	;
										((BOX *)addr)->high.y = t	;
										p1 = strtok(NULL,",(")	;
										sscanf(p1,"%lf",&t)	;
										((BOX *)addr)->low.x = t	;
										p1 = strtok(NULL,",(")	;
										sscanf(p1,"%lf",&t)	;
										((BOX *)addr)->low.y = t	;
								}
								else	{
									((BOX *)addr)->high.x = NAN	;
									((BOX *)addr)->high.y = NAN	;
									((BOX *)addr)->low.x = NAN	;
									((BOX *)addr)->low.y = NAN	;
								}
						}
						break	;
						
					case bindItem_Circle :
						if (_debugMode)
							NSAssert(CIRCLEOID == type,bindTypeError)	;
							
						if (binary)	{
							CIRCLE *p = (CIRCLE *)PQgetvalue(forResult, rowIndex, field);
							
							if (p && !PQgetisnull(forResult, rowIndex, field))
								*(CIRCLE *)addr = *p	;
							else	{
								((CIRCLE *)addr)->center.x = NAN	;
								((CIRCLE *)addr)->center.y = NAN	;
								((CIRCLE *)addr)->radius = NAN	;
							}
						}
						else	{	// (x1,y1),(x2,y2)
							char	*p = PQgetvalue(forResult, rowIndex, field)	;

								if (p && *p)	{
									char	*p1	;
									double	t	;
									
										p1 = strtok(p,",<(")	;
										sscanf(p1,"%lf",&t)	;
										((CIRCLE *)addr)->center.x = t	;
										p1 = strtok(NULL,",<(")	;
										sscanf(p1,"%lf",&t)	;
										((CIRCLE *)addr)->center.y = t	;
										p1 = strtok(NULL,",<(")	;
										sscanf(p1,"%lf",&t)	;
										((CIRCLE *)addr)->radius = t	;
								}
								else	{
									((CIRCLE *)addr)->center.x = NAN	;
									((CIRCLE *)addr)->center.y = NAN	;
									((CIRCLE *)addr)->radius = NAN	;
								}
						}
						break	;
					
					case bindItem_Point :
						if (_debugMode)
							NSAssert(POINTOID == type,bindTypeError)	;
							
						if (binary)	{
							POINT *p = (POINT *)PQgetvalue(forResult, rowIndex, field);
							
							if (p && !PQgetisnull(forResult, rowIndex, field))
								*(POINT *)addr = *p	;
							else	{
								((POINT *)addr)->x = NAN	;
								((POINT *)addr)->y = NAN	;
							}
						}
						else	{	// (x1,y1),(x2,y2)
							char	*p = PQgetvalue(forResult, rowIndex, field)	;

								if (p && *p)	{
									char	*p1	;
									double	t	;
									
										p1 = strtok(p,",(")	;
										sscanf(p1,"%lf",&t)	;
										((POINT *)addr)->x = t	;
										p1 = strtok(NULL,",(")	;
										sscanf(p1,"%lf",&t)	;
										((POINT *)addr)->y = t	;
								}
								else	{
									((POINT *)addr)->x = NAN	;
									((POINT *)addr)->y = NAN	;
								}
						}
						break	;
					
					case	bindItem_Path	:
						if (_debugMode)
							NSAssert(PATHOID == type,bindTypeError)	;

						if (binary)	{
							int		plen = PQgetlength(forResult, rowIndex, field)	;
							char	*p = PQgetvalue(forResult, rowIndex, field)	;	
							PATH	*pval = nil	;
							
								if (p && !PQgetisnull(forResult, rowIndex, field))	{
									pval = (PATH *)malloc(plen + VARHDRSZ)		;
									if (pval != nil)	{
										pval->size = plen	;
										memmove((char *)&pval->npts, p, plen)	;
									}
								}
								*(PATH **)addr = pval	;
						}
						else	{	// ((x,y),(x,y),...,(x,y)) closed path [(x,y),(x,y),...,(x,y)] open path
							char	*p = PQgetvalue(forResult, rowIndex, field)	;
								
								if (p && *p)	{
									double	x, y	;
									char	*p1	;
									int	n = -1	;
									PATH	*pval		;
									size_t	size	;
									BOOL	closed	;
									
									p1 = p	;
									closed = (*p1 == '(')	;
									
									++p1	;
									while (*p1)	{
										if (*p1 == '(')	++n	;
										++p1	;
									}
									
									size = sizeof(PATH) + (sizeof(POINT) * n)	;
									pval = (PATH *)malloc(size)		;
									if (pval != NULL)	{
										pval->npts = n	;

										n = 0	;
										p1 = strtok(p,",(")	;
										while (p1 != NULL)	{
											sscanf(p1,"%lf",&x)	;
											p1 = strtok(NULL,",(");
											sscanf(p1,"%lf",&y)	;
											pval->p[n].x = x	;
											pval->p[n].y = y	;
											++n	;
											p1 = strtok(NULL,",(");
										}
										pval->size = size	;
										if (n <= 1)
											pval->closed = 0	;
										else
											pval->closed = closed	;
									}
									
									*(PATH **)addr = pval	;
								}
								else
									*(PATH **)addr = NULL	;
						}
						break	;

					case bindItem_INet :
						if (_debugMode)
							NSAssert(INETOID == type,bindTypeError)	;

						if (binary)	{
							int		i, plen = PQgetlength(forResult, rowIndex, field)	;
							unsigned char *p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
							
							if (p && !PQgetisnull(forResult, rowIndex, field))
								memmove((unsigned char *)addr, p, plen)	;
							else
								for(i=0, p = (unsigned char *)addr;i<plen;i++,p++)
									*p = 0	;
						}
						else	{
							//	a0.a1.a2.a3/mask => 2 mask 0 0 a0 a1 a2 a3 0 0 0 0
							unsigned char *p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
								
							if (p && *p)	{
								int a0, a1, a2, a3,m	;
								
								sscanf(p,"%d.%d.%d.%d/%d",&a0,&a1,&a2,&a3,&m)	;
								p = (unsigned char *)addr	;
								p[0] = 2	;
								p[1] = m	;
								p[2] = 0	;
								p[3] = 0	;
								p[4] = (unsigned char )a0	;
								p[5] = (unsigned char )a1	;
								p[6] = (unsigned char )a2	;
								p[7] = (unsigned char )a3	;
								p[8] = (unsigned char )0	;
								p[9] = (unsigned char )0	;
								p[10] = (unsigned char )0	;
								p[11] = (unsigned char )0	;
							}
							else	{
								int	i	;
								
								for(i=0, p = (unsigned char *)addr;i<12;i++,p++)
									*p = 0	;
							}
						}
						break	;

					case bindItem_Cidr :
						if (_debugMode)
							NSAssert(CIDROID == type,bindTypeError)	;

						if (binary)	{
							int		i, plen = PQgetlength(forResult, rowIndex, field)	;
							unsigned char *p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
							
							if (p && !PQgetisnull(forResult, rowIndex, field))
								memmove((unsigned char *)addr, p, plen)	;
							else
								for(i=0, p = (unsigned char *)addr;i<plen;i++,p++)
									*p = 0	;
						}
						else	{
							//	a0.a1.a2.a3/mask => 2 mask 1 0 a0 a1 a2 a3 0 0 0 0
							unsigned char *p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
								
							if (p && *p)	{
								int a0, a1, a2, a3,m	;
								
								sscanf(p,"%d.%d.%d.%d/%d",&a0,&a1,&a2,&a3,&m)	;
								p = (unsigned char *)addr	;
								p[0] = 2	;
								p[1] = m	;
								p[2] = 1	;
								p[3] = 0	;
								p[4] = (unsigned char )a0	;
								p[5] = (unsigned char )a1	;
								p[6] = (unsigned char )a2	;
								p[7] = (unsigned char )a3	;
								p[8] = (unsigned char )0	;
								p[9] = (unsigned char )0	;
								p[10] = (unsigned char )0	;
								p[11] = (unsigned char )0	;
							}
							else	{
								int	i	;
								
								for(i=0, p = (unsigned char *)addr;i<12;i++,p++)
									*p = 0	;
							}
						}
						break	;

					case bindItem_MACAddr :
						if (_debugMode)
							NSAssert(MACADDROID == type,bindTypeError)	;

						if (binary)	{
							int		i, plen = PQgetlength(forResult, rowIndex, field)	;
							unsigned char *p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
							
							if (p && !PQgetisnull(forResult, rowIndex, field))
								memmove((unsigned char *)addr, p, plen)	;
							else
								for(i=0, p = (unsigned char *)addr;i<plen;i++,p++)
									*p = 0	;
						}
						else	{
							//	a0.a1.a2.a3/mask => 2 mask 1 0 a0 a1 a2 a3 0 0 0 6
							unsigned char *p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
								
							if (p && *p)	{
								int a0, a1, a2, a3, a4, a5	;
								
								sscanf(p,"%x:%x:%x:%x:%x:%x",&a0,&a1,&a2,&a3,&a4,&a5)	;
								p = (unsigned char *)addr	;
								p[0] = (unsigned char )a0	;
								p[1] = (unsigned char )a1	;
								p[2] = (unsigned char )a2	;
								p[3] = (unsigned char )a3	;
								p[4] = (unsigned char )a4	;
								p[5] = (unsigned char )a5	;
							}
							else	{
								int	i	;
								
								for(i=0, p = (unsigned char *)addr;i<6;i++,p++)
									*p = 0	;
							}
						}
						break	;
					
					case	bindItem_BIT	:	{
						if (_debugMode)	{
							NSAssert( ((ZPBITOID == type) || (VARBITOID == type)), bindTypeError)	;
						}
						
						int		i, plen = PQgetlength(forResult, rowIndex, field)	;
						unsigned char 	*p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
						BOOL	*pval = NULL, *pval1	;
						
							if (p && !PQgetisnull(forResult, rowIndex, field))	{
								if (binary)	{ 
									int	nBits = *(int *)p	;
									
									p += sizeof(int)	;
									pval = (BOOL *)malloc(nBits * sizeof(BOOL))	;
									if (pval)	{
										int	nLoops = nBits / 8	;
										unsigned char *p1 = p, *pval1 = pval, mask	;
										
											while (nLoops--)	{
												*pval1++ = ((*p1 & 0x80) != 0)	;
												*pval1++ = ((*p1 & 0x40) != 0)	;
												*pval1++ = ((*p1 & 0x20) != 0)	;
												*pval1++ = ((*p1 & 0x10) != 0)	;
												*pval1++ = ((*p1 & 0x08) != 0)	;
												*pval1++ = ((*p1 & 0x04) != 0)	;
												*pval1++ = ((*p1 & 0x02) != 0)	;
												*pval1++ = ((*p1 & 0x01) != 0)	;
												++p1	;
											}
											nLoops = nBits % 8	;
											mask = 0x80	;
											while (nLoops--)	{
												*pval1++ = ((*p1 & mask) != 0)	;
												mask >>= 1	;
											}
											((BOOLARRAY *)addr)->size = nBits	;
											((BOOLARRAY *)addr)->bits = pval	;
									}
								}
								else	{
									pval = (BOOL *)malloc( plen * sizeof(BOOL) )	;
									if (pval)	{
										pval1 = pval;	
										for(i=0;i<plen;i++,pval1++,p++)
											*pval1 = (*p == '1')	;
										((BOOLARRAY *)addr)->size = plen;
										((BOOLARRAY *)addr)->bits = pval;
									}	
								}
							}
							if (pval == NULL)	{
								((BOOLARRAY *)addr)->size = 0	;
								((BOOLARRAY *)addr)->bits = NULL;
							}
						}
						break	;
					
					case	bindItem_NSArray	:	{
							if (_debugMode)	{
								NSAssert( 
									(BOOL_ARRAY_OID == type) || 
									//	(BYTEA_ARRAY_OID == type) || 
									(CHAR_ARRAY_OID == type) || 
									(SHORT_ARRAY_OID == type) || 
									(INT_ARRAY_OID == type) || 
									//	(TEXT_ARRAY_OID == type) || 
									(OID_ARRAY_OID == type) || 
									(STRING_ARRAY_OID == type) || 
									(VSTRING_ARRAY_OID == type) || 
									(LONGLONG_ARRAY_OID == type) || 
									//	(POINT_ARRAY_OID == type) || 
									//	(LSEG_ARRAY_OID == type) || 
									//	(PATH_ARRAY_OID == type) || 
									//	(BOX_ARRAY_OID == type) || 
									(FLOAT_ARRAY_OID == type) || 
									(DOUBLE_ARRAY_OID == type),	// || 
									//(POLYGON_ARRAY_OID == type), 
									bindTypeError)	;
							}
							
							*(NSArray **)addr = nil	;

							if (PQgetisnull(forResult, rowIndex, field))
								break	;
								
							/*
							if (type == STRING_ARRAY_OID)	{
								dumpHexa(PQgetvalue(forResult, rowIndex, field),PQgetlength(forResult, rowIndex, field))	;
								
								break	;
							}
							*/
							
							if (binary)	{
								unsigned char *p = (unsigned char *)PQgetvalue(forResult, rowIndex, field);
								
								int	*iPtr = (int *)p	;
								int	nDims = iPtr[0]		;
								int	*dims = iPtr + 2	;
								int	*dimLowers = dims + nDims	;
								void *dataPtr= (void *)(dimLowers + nDims)	;
								NSMutableArray *array = [NSMutableArray arrayWithCapacity:dims[0]]	;
								
								fillLevel(array, 0, dims, nDims, type, &dataPtr)	;
								
								*(NSArray **)addr = array	;
							}
							else	{
								char 	*p = (char *)PQgetvalue(forResult, rowIndex, field);
								char	*p1	;
								int		i, dims[100], curDim = -1, nDims = -1	;
								BOOL	inString = NO	;
								
									for(i=0;i<100;i++) dims[i] = 1	;
									
									p1 = p	;
									while(*p1)	{
										if (*p1 == '"')
											inString = !inString	;
										
										if (inString)	{
											if (*p1 == '\\')
												++p1	;
										}
										else	{
											if (*p1 == '{')	{
												++curDim	;
												if (curDim > nDims) nDims = curDim	;
											}
											else if (*p1 == '}')
												--curDim	;
											else if (*p1 == ',')
												dims[curDim]++	;
										}
											
										++p1	;
									}
									if (nDims != 0)
										dims[nDims] = (dims[nDims] + dims[0] - 1) / dims[0]	;
									++nDims	;
									
									/*
									printf("%d Dimensions ",nDims)	;
									for(i=0;i<nDims;i++)
										printf("%d ",dims[i])	;
									printf("\n")	;
									*/
									
									NSMutableArray	*array = [NSMutableArray arrayWithCapacity:dims[0]]	;

									p1 = p;
									while (*p1 == '{') ++p1	;
									
									fillArray(array, 0, dims, nDims, type, &p1)	;
									
									*(NSArray **)addr = array	;
							}
						}
						break;
							
					default	:
						NSLog(@"PostgreSQL FrameWork internal error: invalid bin object type")	;
						break	;
				}
		}
}

-(id)curRowField:(int)fieldIndex
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
			NSAssert((_curRowIndex >= 0) && (_curRowIndex < _rowsInResult), invalidRowIndex)	;
		}
		
		return [self _fetchFieldValue:_lastResult atIndex:_curRowIndex forField:fieldIndex binary:PQbinaryTuples(_lastResult)]	;
}

-(NSArray *)resultRowAsArray
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
			NSAssert((_curRowIndex >= 0) && (_curRowIndex < _rowsInResult), invalidRowIndex)	;
		}

		
	int				numFields = [self _fetchNFields]	;
	NSMutableArray	*row = [NSMutableArray arrayWithCapacity:numFields]	;
	int				binary = PQbinaryTuples(_lastResult)	;
	int				fieldIdx	;
	
		for( fieldIdx = 0 ; fieldIdx < numFields ; fieldIdx++)	{
			[row addObject:[self _fetchFieldValue:_lastResult atIndex:_curRowIndex forField:fieldIdx binary:binary]]	;
		}
	
		return row	;
}

-(NSDictionary *)resultAsDictionary
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

    int				numFields = [self _fetchNFields], numRows = _rowsInResult;
    NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithCapacity: 2];
	int		fieldIdx, oldCurRowIndex	;

	NSMutableArray	*columns = [NSMutableArray arrayWithCapacity:numFields]	;
	NSMutableArray	*rows = [NSMutableArray arrayWithCapacity:numFields]	;
	int	binary = PQbinaryTuples(_lastResult)	;
	
#if	DEMO_MODE
		if (numRows > MAX_RESULTS)	numRows = MAX_RESULTS	;
#endif

		for( fieldIdx = 0 ; fieldIdx < numFields ; fieldIdx++)	{
			NSMutableDictionary *columnDict = [NSMutableDictionary dictionaryWithCapacity: 3];
			
				[columnDict setObject:[NSString stringWithCString: [self _fetchFieldName:fieldIdx]] forKey:PgSQLNameKeyField];
				[columnDict setObject:[NSNumber numberWithInt: [self _fetchFieldType:fieldIdx]] forKey:PgSQLTypeKeyField];
				[columnDict setObject:[NSString stringWithCString: [self _fetchFieldTypeName:fieldIdx]] forKey:PgSQLTypeNameKeyField];
			
				[columns addObject:columnDict]	;
		}
			
		oldCurRowIndex = _curRowIndex	;
		for(_curRowIndex = 0; _curRowIndex < numRows; _curRowIndex++)	{
			NSMutableArray	*fields = [NSMutableArray arrayWithCapacity:numFields]	;
		
				for( fieldIdx = 0 ; fieldIdx < numFields ; fieldIdx++)	{
					[fields addObject:[self _fetchFieldValue:_lastResult atIndex:_curRowIndex forField:fieldIdx binary:binary]]	;
				}
				[rows addObject:fields]	;
        }
		_curRowIndex = oldCurRowIndex	;

		[dict setObject:columns forKey:PgSQLColumnsKeyField];
        [dict setObject:rows forKey:PgSQLRowsKeyField];

		return dict	;
}

-(BOOL)nextRow
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

		++_curRowIndex	;
		if (_curRowIndex < _rowsInResult)	{
			[self _fetchData]	;
			return YES	;
		}
#if	DEMO_MODE
		if (_curRowIndex >= MAX_RESULTS)	_curRowIndex = MAX_RESULTS	;
#else
		_curRowIndex = _rowsInResult	;
#endif
		return NO	;	
}

-(BOOL)previousRow
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}

		--_curRowIndex	;
		if (_curRowIndex >= 0)	{
			[self _fetchData]	;
			return YES	;
		}
		_curRowIndex = -1	;
		return NO;
}

-(void)afterLastRow
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}
#if	DEMO_MODE
		if (_curRowIndex >= MAX_RESULTS)	_curRowIndex = MAX_RESULTS	;
#else
		_curRowIndex = _rowsInResult	;
#endif
}

-(void)beforeFirstRow
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}
		_curRowIndex = -1	;
}

-(void)firstRow
{
		[self goRow:0]	;
}

-(void)lastRow
{
		[self goRow:_rowsInResult-1]	;
}

-(void)goRow:(int)rowIndex
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}
#if	DEMO_MODE
		if (rowIndex >= MAX_RESULTS)	rowIndex = MAX_RESULTS	;
#endif
		if (rowIndex < 0)	{
			rowIndex = -1	;
		}
		if (rowIndex >= _rowsInResult)	{
			rowIndex = _rowsInResult	;
		}
		_curRowIndex = rowIndex	;
}

-(void)goRowRelative:(int)deltaIndex
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
			NSAssert(_lastResult != nil, nilLastResultMsg)	;
		}
	int	rowIndex = _curRowIndex + deltaIndex	;
#if	DEMO_MODE
		if (rowIndex >= MAX_RESULTS)	rowIndex = MAX_RESULTS	;
#endif
		if (rowIndex < 0)	{
			rowIndex = -1	;
		}
		if (rowIndex >= _rowsInResult)	{
			rowIndex = _rowsInResult	;
		}
		_curRowIndex = rowIndex	;
}

-(int)resultColumnCount
{
		return [self _fetchNFields]	;
}

-(const char *)resultColumnName:(int)col
{
		return [self _fetchFieldName:col]	;
}

-(const char *)resultColumnTypeName:(int)col
{
		return [self _fetchFieldTypeName:col]	;
}

-(Oid)resultColumnType:(int)col
{
		return [self _fetchFieldType:col]	;
}

-(BOOL)resultReturned
{
		return [self rowsAffected] > 0	;
}

-(const char *)resultTableName:(int)col
{
		return "NOT_SUPPORTED_BY_POSTGRESQL"	;
}

-(const char *)uniqueRowIdForTable:(const char *)tableName
{
	char		command[512]	;
	char		*res = "0"		;
	PGresult 	*result	;
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		sprintf(command,"SELECT max(_rowid)+1 FROM %s",tableName)	;
		result = PQexec(_conn,command)	;
		if (PQntuples(result) > 0)	{
			res = PQgetvalue(result, 0, 0)	;
			PQclear(result)	;
		}
			
		return res	;
}

-(const char *)uniqueRowIdForTable:(const char *)tableName column:(const char *)colName
{
	char		command[512]	;
	char		*res = "0"		;
	PGresult 	*result	;
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		sprintf(command,"SELECT max(%s)+1 FROM %s",colName,tableName)	;
		result = PQexec(_conn,command)	;
		if (PQntuples(result) > 0)	{
			res = PQgetvalue(result, 0, 0)	;
			PQclear(result)	;
		}
			
		return res	;
}

-(PgSQLListener *)startNotificationFor:(const char *)inTableName delegate:(id)notificatonDelegate userInfo:(id)userInfo
{
	char	command[256]	;
	PGresult	*result	;
	PgSQLListener	*listener = nil	;
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		listener = (PgSQLListener *)[_listeners objectForKey:[NSString stringWithCString:inTableName]];

		if (listener != NULL)	{
			[listener setToBeNotified:notificatonDelegate]	;
			return listener	;
		}
		
		sprintf(command,"LISTEN %s",inTableName)	;
		result = PQexec(_conn,command)	;
		if (RESULT_ISOK(result))	{
			NSMutableDictionary *aDict = [NSMutableDictionary dictionaryWithCapacity:3]	;
			
			if (userInfo == nil)
				userInfo = [NSNull null]	;
			[aDict setObject:userInfo forKey:PgSQLuserInfoNotificationField]	;
			[aDict setObject:[NSString stringWithCString:inTableName] forKey:PgSQLconditionNotificationField]	;
			[aDict setObject:self forKey:PgSQLconnectionNotificationField]	;

			listener = [PgSQLListener newPgSQLListener:inTableName notify:notificatonDelegate userInfo:aDict];
			[_listeners setObject:listener forKey:[NSString stringWithCString:inTableName]];
		}
		PQclear(result)	;
		return listener	;
}

-(void)removeNotificationFor:(PgSQLListener *)listener
{
		[_listeners removeObjectForKey:[listener getCondition]]	;
}

#define	BUFSIZE	1024

-(NSData *)retrieveBinary:(Oid)inOid
{
	int	lobj_fd	;
	NSMutableData	*result	;
	char	buffer[BUFSIZE]	;
	int		nbytes	;
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		lobj_fd  = lo_open(_conn,inOid, INV_READ)	;
		if (lobj_fd < 0)	{
			return nil	;
		}
		
		lo_lseek(_conn, lobj_fd, 0, SEEK_SET)	;
		result = [NSMutableData data]	;
		
		while ((nbytes = lo_read(_conn, lobj_fd, buffer, BUFSIZE)) > 0)	{
			[result appendBytes:buffer length:nbytes]	;
		}
		lo_close(_conn, lobj_fd)	;
		
		return result	;
}

-(NSData *)retrieveBinaryFromStartLength:(Oid)inOid start:(int)inStart length:(int)length
{
	int	lobj_fd	;
	NSMutableData	*result	;
	char	*buffer	;
	int		nbytes	;
	
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		buffer = malloc(length)	;
		if (buffer == nil)
			return nil	;
			
		lobj_fd  = lo_open(_conn,inOid, INV_READ)	;
		if (lobj_fd < 0)	{
			return nil	;
		}
		
		lo_lseek(_conn, lobj_fd, inStart, SEEK_SET)	;
		
		nbytes = lo_read(_conn, lobj_fd, buffer, length)	;				
		result = [NSMutableData dataWithBytes:buffer length:nbytes]	;

		lo_close(_conn, lobj_fd)	;
		
		free(buffer)	;
		return result	;
}

-(Oid)insertBinary:(unsigned char *)inData size:(int)size
{
	Oid		lobjId	;
	int		lobj_fd	;
	unsigned char *bPtr	;
	int		nbytes, nLoops, tmp	;

		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		lobjId = lo_creat(_conn, INV_READ | INV_WRITE)	;
		if (lobjId == 0)
			return 0	;
			
		lobj_fd = lo_open(_conn, lobjId, INV_WRITE)	;
		
		nLoops = size / BUFSIZE	;
		bPtr = inData	;
		
		while (nLoops--)	{
			tmp = lo_write(_conn, lobj_fd, bPtr, BUFSIZE)	;
			if (tmp < BUFSIZE)	{
				lo_close(_conn, lobj_fd)	;
				lo_unlink(_conn, lobjId)	;
				return 0	;
			}
			bPtr += BUFSIZE	;
		}
		
		nbytes = size % BUFSIZE	;
		tmp = lo_write(_conn, lobj_fd, bPtr, nbytes)	;
		if (tmp < nbytes)	{
			lo_close(_conn, lobj_fd)	;
			lo_unlink(_conn, lobjId)	;
			return 0	;
		}
		
		lo_close(_conn, lobj_fd)	;
		return lobjId	;
}

-(Oid)insertBinaryFromFile:(NSFileHandle *)inFile
{
	Oid		lobjId	;
	int		lobj_fd	;
	char	buffer[BUFSIZE]	;
	int		nbytes, tmp, fd	;

		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		fd = [inFile fileDescriptor]	;
		
		lobjId = lo_creat(_conn, INV_READ | INV_WRITE)	;
		if (lobjId == 0)
			return 0	;
			
		lobj_fd = lo_open(_conn, lobjId, INV_WRITE)	;
		
		while ((nbytes = read(fd, buffer, BUFSIZE)) > 0)	{
			tmp = lo_write(_conn, lobj_fd, buffer, BUFSIZE)	;
			if (tmp < nbytes)	{
				lo_close(_conn, lobj_fd)	;
				lo_unlink(_conn, lobjId)	;
				return 0	;
			}
		}
		
		lo_close(_conn, lobj_fd)	;
		
		[inFile closeFile]	;
		
		return lobjId	;
}

-(Oid)insertBinaryFromData:(NSData *)inData
{
	Oid		lobjId	;
	int		lobj_fd	;
	const char	*bufPtr	;
	int		nbytes, tmp	;

		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		lobjId = lo_creat(_conn, INV_READ | INV_WRITE)	;
		if (lobjId == 0)
			return 0	;
			
		lobj_fd = lo_open(_conn, lobjId, INV_WRITE)	;
		
		bufPtr = [inData bytes]	;
		nbytes = [inData length]	;
		
		tmp = lo_write(_conn, lobj_fd,(char *) bufPtr, nbytes)	;
		if (tmp < nbytes)	{
			lo_close(_conn, lobj_fd)	;
			lo_unlink(_conn, lobjId)	;
			return 0	;
		}
		
		lo_close(_conn, lobj_fd)	;
				
		return lobjId	;
}

-(BOOL)overWriteBinaryWithDataFromStart:(Oid)inOid data:(NSData *)inData start:(int)inStart
{
	int		lobj_fd	;
	const char	*bufPtr	;
	int		nbytes, tmp	;

		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		lobj_fd = lo_open(_conn, inOid, INV_READ)	;
		lo_lseek(_conn, lobj_fd, inStart, SEEK_SET)	;
		
		bufPtr = [inData bytes]		;
		nbytes = [inData length]	;
		
		tmp = lo_write(_conn, lobj_fd,(char *) bufPtr, nbytes)	;
		
		lo_close(_conn, lobj_fd)	;
				
		return (tmp == nbytes)	;
}

-(int)exportBinaryToFile:(Oid)inOid pathName:(const char *)inPathName
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		return lo_export(_conn, inOid, inPathName)	;
}

-(Oid)importBinaryFromFile:(const char *)inPathName
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		return lo_import(_conn, inPathName)	;
}

-(int)unlinkBinary:(Oid)inOid
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		return lo_unlink(_conn, inOid)	;
}

-(BOOL)asyncResultAvailable
{
		return	_asyncCommandResultAvailable	;
}

-(long)getTimeOut
{
		return _timeOut	;
}

-(void)setTimeOut:(long)inTimeOut
{
		_timeOut = inTimeOut	;
}

-(void)_setAsyncThread:(PgASyncThread *)inAsyncThread
{
		_asyncThread = [inAsyncThread retain]	;
}

-(void)_resultAvailable
{
		_asyncCommandResultAvailable = YES	;
		CLEAR_LASTRESULT;
		_lastResult = PQgetResult(_conn)	;
		_rowsInResult = PQntuples(_lastResult)	;
		_fieldsInResult = PQnfields(_lastResult)	;

		[[NSNotificationCenter defaultCenter] postNotificationName:PostgreSQLResultAvailable object:nil userInfo:nil]	;
		
		if (_delegate != nil)	{
			if ( [_delegate respondsToSelector:@selector(postgreSQLAsyncResultAvailable:)] )
				objc_msgSend(_delegate, @selector(postgreSQLAsyncResultAvailable:), self)	;
		}
}

-(void)_timeOut
{
		[[NSNotificationCenter defaultCenter] postNotificationName:PostgreSQLTimeOut object:nil userInfo:nil]	;

		if (_delegate != nil)	{
			if ( [_delegate respondsToSelector:@selector(postgreSQLAsyncResulTimedOut:)] )
				objc_msgSend(_delegate, @selector(postgreSQLAsyncResulTimedOut:), self)	;
		}
}

-(void)encodeWithCoder:(NSCoder *)coder
{
		[coder encodeObject:_uniqueKeyID]	; 
		[coder encodeValueOfObjCType:@encode(BOOL) at:&_debugMode]	; 
		[coder encodeValueOfObjCType:@encode(long) at:&_timeOut]	; 
		[coder encodeObject:_delegate]	; 
		[coder encodeObject:_localUSDict]	; 
	
	BOOL	connected = (_conn != nil);
		[coder encodeValueOfObjCType:@encode(BOOL) at:&connected]	; 
		if (connected)	{
			[coder encodeObject:[NSString stringWithCString:[self databaseName]]]	; 
			[coder encodeObject:[NSString stringWithCString:[self hostName]]]	; 
			[coder encodeObject:[NSString stringWithCString:[self loginName]]]	; 
			[coder encodeObject:[NSString stringWithCString:[self password]]]	; 
			[coder encodeObject:[NSString stringWithCString:[self tty]]]	; 
			[coder encodeObject:[NSString stringWithCString:[self options]]]	; 
		}
}

-(id)initWithCoder:(NSCoder *)coder
{
		self  = [super init]	;
		_conn = nil		;
		_traceFile = NULL	;
		_lastResult = nil	;
		_cmdBuffer = [[NSMutableString string] retain]	;
		_listeners = [[NSMutableDictionary dictionaryWithCapacity:10] retain]	;
		_commandExecuted = NO	;
		_asyncCommandInProgress = NO	;
		_asyncCommandResultAvailable = NO	;
		_bindingArray = nil	;
		_transactionLevel = 0	;
		_listeningTimer = [[NSTimer scheduledTimerWithTimeInterval:1.0 target:self selector:@selector(handlePgNotify:) userInfo:nil repeats:YES] retain];
		_asyncThread = nil	;
	
		[[NSNotificationCenter defaultCenter] 
			addObserver:self
			selector:@selector(handleAppWillTermNotification:)
			name:NSApplicationWillTerminateNotification
			object:nil]	;

		_uniqueKeyID = [[coder decodeObject] retain]	; 
		[coder decodeValueOfObjCType:@encode(BOOL) at:&_debugMode]	;
		[coder decodeValueOfObjCType:@encode(long) at:&_timeOut]	; 
		_delegate = [[coder decodeObject] retain]	; 
		_localUSDict = [[coder decodeObject] retain]	; 

	BOOL connected	;
	
		[coder decodeValueOfObjCType:@encode(BOOL) at:&connected]	;

		if (connected)	{
			NSString *databaseName = [coder decodeObject]	;
			NSString *hostName = [coder decodeObject]	;
			NSString *loginName = [coder decodeObject]	;
			NSString *password = [coder decodeObject]	;
			NSString *tty = [coder decodeObject]	;
			NSString *options = [coder decodeObject]	;
			
			NSMutableString *connInfo = [NSMutableString string]	;
			
				if ([hostName length])	{
					[connInfo appendString:@"host='"]	;
					[connInfo appendString:hostName]	;
					[connInfo appendString:@"'"]	;
				}
			
				if ([databaseName length])	{
					[connInfo appendString:@"dbname='"]	;
					[connInfo appendString:databaseName]	;
					[connInfo appendString:@"'"]	;
				}
			
				if ([loginName length])	{
					[connInfo appendString:@"user='"]	;
					[connInfo appendString:loginName]	;
					[connInfo appendString:@"'"]	;
				}
			
				if ([password length])	{
					[connInfo appendString:@"password='"]	;
					[connInfo appendString:password]	;
					[connInfo appendString:@"'"]	;
				}
			
				if ([tty length])	{
					[connInfo appendString:@"tty='"]	;
					[connInfo appendString:tty]	;
					[connInfo appendString:@"'"]	;
				}
			
				if ([options length])	{
					[connInfo appendString:@"options='"]	;
					[connInfo appendString:options]	;
					[connInfo appendString:@"'"]	;
				}
				
			int	errCode;
				[self connectToDatabase:[connInfo cString] return:&errCode]	;
		}
		
		return self	;
}

-(PQnoticeProcessor)setNoticeProcessor:(PQnoticeProcessor)proc userArg:(void *)arg
{
		if (_debugMode)	{
			NSAssert(_conn != nil, nilConnectionMsg)	;
		}

		return PQsetNoticeProcessor(_conn, proc, arg);
}

static void showAllNotiveProcessor(void *arg,const char *message)
{
	NSMutableDictionary *dict = (NSMutableDictionary *)arg	; 
	const char	*s = message + 8	;	// 8 == strlen("NOTICE: ")	
		
	char *s1 = strstr(s," is ")	;
	
		if (s1 != NULL)	{
			*s1 = '\0'	;
			NSString	*key = [NSString stringWithCString:s+1]	;
			s1 += 4	;	// 4 = strlen(" is ")	
			NSString	*value = [NSString stringWithCString:s1]	;
			
			[dict setObject:value forKey:key]	;
		}
		else
			NSLog(@"error processing %s",message)	;
}

-(NSDictionary *)_buildVariablesDictionary
{
		if (_debugMode)
			NSAssert(_conn != nil, nilConnectionMsg)	;
			
		if (_asyncCommandInProgress)	{
			NSLog(@"executeAsyncCommand in progress")	;
			return nil	;
		}

	NSMutableDictionary	*dict = [NSMutableDictionary dictionary]	;
	PQnoticeProcessor oldNotice = [self setNoticeProcessor:showAllNotiveProcessor userArg:dict];
	PGresult	*result = PQexec(_conn,"SHOW ALL")	;
		PQclear(result)	;
		[self setNoticeProcessor:oldNotice userArg:nil];
		return dict	;
}

static NSDictionary	*_variablesDictionary = nil	;

-(NSDictionary *)getVariables
{

		if (_variablesDictionary != nil)
			return _variablesDictionary	;
			
		return _variablesDictionary = [[self _buildVariablesDictionary] retain]	;
}

-(NSDictionary *)loadVariables
{
		if (_variablesDictionary != nil)
			[_variablesDictionary release]	;
			
		return _variablesDictionary = [[self _buildVariablesDictionary] retain]	;
}

@end
