/*-------------------------------------------------------------------------
 *
 * copyto.c
 *	  An implementation of DestReceiver that stores the result tuples in
 *	  a delimited file
 *
 *
 * Portions Copyright (c) 2003, Alcove Systems Engineering
 * Portions Copyright (c) 1996-2002, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 * IDENTIFICATION
 *	  $Header: /usr/local/cvsroot/pgsql-server/src/backend/executor/copytoReceiver.c,v 1.1 2003/03/27 16:53:15 momjian Exp $
 *
 *-------------------------------------------------------------------------
 */

#include "postgres.h"
#include "miscadmin.h"
#include "utils/memutils.h"
#include "utils/lsyscache.h"
#include "access/heapam.h"
#include "commands/copy.h"

#include "executor/copyto.h"

typedef struct
{
	DestReceiver	pub;
	char		   *delim;
	char		   *null_print;

	/*
	 * this info is saved at receiver setup time so that we don't
	 * need to regenerate for each tuple
	 */

	FmgrInfo	   *out_functions;
	Oid			   *elements;
	bool		   *isvarlena;
	List		   *attnums;

	MemoryContext	cxt;
} CopyToState;

/*
 * Receive a tuple from the executor and store it a file.
 */

static void
copytoSetupReceiver(DestReceiver *self, int operation, TupleDesc typeinfo)
{
	CopyToState			*myState = (CopyToState *)self;
	Form_pg_attribute 	*attr;
	int					num_phys_attrs,
						i;
	FmgrInfo			*out_functions;
	Oid					*elements;
	bool				*isvarlena;
	List				*attnums = NIL;

	if (operation != CMD_SELECT)
		elog(ERROR, "Unexpected operation type: %d", operation);

	attr = typeinfo->attrs;
	num_phys_attrs = typeinfo->natts;

	/*
	 * Get info about the columns we need to process.
	 *
	 * +1's here are to avoid palloc(0) in a zero-column table.
	 */

	out_functions = (FmgrInfo *) palloc((num_phys_attrs + 1) * 
													sizeof(FmgrInfo));
	elements = (Oid *) palloc((num_phys_attrs + 1) * sizeof(Oid));
	isvarlena = (bool *) palloc((num_phys_attrs + 1) * sizeof(bool));

	for(i=0;i<num_phys_attrs;i++) {
		Oid			out_func_oid;

		if(attr[i]->attisdropped)
			continue;

		getTypeOutputInfo(attr[i]->atttypid,
							  &out_func_oid, &elements[i],
							  &isvarlena[i]);
		fmgr_info(out_func_oid, &out_functions[i]);
		attnums = lappendi(attnums,i + 1);
	}
	myState->attnums = attnums;
	myState->out_functions = out_functions;
	myState->elements = elements;
	myState->isvarlena = isvarlena;
}

static void
copytoReceiveTuple(HeapTuple tuple, TupleDesc typeinfo, DestReceiver *self)
{
	CopyToState *myState = (CopyToState *) self;
	char	*delim = myState->delim;
	int		attnum;
	char	*null_print = myState->null_print;
	MemoryContext oldcontext;
	FmgrInfo   *out_functions = myState->out_functions;
	Oid		*elements = myState->elements;
	char	   *string;
	List	   *cur,
			   *attnums = myState->attnums;
	bool		need_delim = false;
	Form_pg_attribute *attr;

	attr = typeinfo->attrs;


	CHECK_FOR_INTERRUPTS();

	oldcontext = MemoryContextSwitchTo(myState->cxt);

	foreach(cur,attnums) {
		Datum		value;
		bool		isnull;
		attnum = lfirsti(cur);

		value = heap_getattr(tuple, attnum, typeinfo, &isnull);

		if (need_delim)
			CopySendChar(delim[0]);

		need_delim = true;


		if (isnull)
		{
			CopySendString(null_print);		/* null indicator */
		}
		else
		{
			char *cp;
			int len;
			string = DatumGetCString(
							FunctionCall3(&out_functions[attnum - 1],value,
							ObjectIdGetDatum(elements[attnum - 1]),
							Int32GetDatum(attr[attnum - 1]->atttypmod)));

			/* informix mod */
			/* right trim */
			len = strlen(string);
			if (len > 0)
			{
				cp = string+len-1;
				while(len--)
				{
					if (isspace(*cp)) *cp-- = '\0'; 
				}
			}
			CopyAttributeOut(string, delim);
		}
	}
	/*
	 * XXX: Informix needs a trailing seperator
	 */
	CopySendChar(delim[0]);
	CopySendChar('\n');

	MemoryContextReset(myState->cxt);
	MemoryContextSwitchTo(oldcontext);
}

static void
copytoShutdownReceiver(DestReceiver *self)
{
	CopyToState	*myState = (CopyToState *)self;
	pfree(myState->isvarlena);
	pfree(myState->elements);
	pfree(myState->out_functions);
	MemoryContextDelete(myState->cxt);
}

static void
copytoDestroyReceiver(DestReceiver *self)
{
	pfree(self);
}

DestReceiver *
copytoReceiverCreateDR(copydat *cd)
{
	CopyToState *self = (CopyToState *) palloc(sizeof(CopyToState));

	self->pub.receiveTuple = copytoReceiveTuple;
	self->pub.rStartup = copytoSetupReceiver;
	self->pub.rShutdown = copytoShutdownReceiver;
	self->pub.rDestroy = copytoDestroyReceiver;
	self->null_print = cd->null_print;
	self->delim = cd->delim;
	self->elements = NULL; 
	self->out_functions = NULL;
	self->isvarlena = NULL;

   	self->cxt = AllocSetContextCreate(CurrentMemoryContext,
									  "UNLOAD",
									  ALLOCSET_DEFAULT_MINSIZE,
									  ALLOCSET_DEFAULT_INITSIZE,
									  ALLOCSET_DEFAULT_MAXSIZE);
	self->attnums = cd->attnums;

	return (DestReceiver *) self;
}
