/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines the main of skroogeconvert.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdocumentbank.h"
#include "skgtransactionmng.h"
#include "skgimportexportmanager.h"
#include "skgtraces.h"

#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <KDE/KLocale>
#include <kurl.h>

#include <QCoreApplication>
#include <QFileInfo>

/**
 * To compute the version
 */
#define VER1_(x) #x
/**
 * To compute the version
 */
#define VER_(x) VER1_(x)
/**
 * To compute the version
 */
#define VER VER_(SKGVERSION)


/**
 * The main of the application
 * @param argc number of arguments
 * @param argv arguments
 * @return return code
 */
int main(int argc, char** argv)
{
    KAboutData about("skroogeconvert",
                     0,
                     ki18nc("The name of the application", "Skrooge Convert"),
                     VER,
                     ki18nc("The description of the application", "A conversion tool for financial files (KMyMoney, GnuCash, Skrooge, ...)"),
                     KAboutData::License_GPL_V3,
                     ki18nc("Fullname", "(c) 2007-%1 Stephane MANKOWSKI & Guillaume DE BURE").subs(QDate::currentDate().toString("yyyy")),
                     KLocalizedString(),
                     "http://skrooge.org");

    about.addAuthor(ki18nc("Fullname", "Stephane MANKOWSKI"), ki18nc("A job description", "Architect & Developer"), "stephane@mankowski.fr", ""
                    , "miraks"
                   );
    about.addAuthor(ki18nc("Fullname", "Guillaume DE BURE"), ki18nc("A job description", "Developer"), "guillaume.debure@gmail.com", ""
                    , "willy9"
                   );
    about.setOtherText(ki18nc("The description of the application", "The application name is inspired by Charles Dicken's tale <i>A Christmas Carol</i>, where the main character, Ebenezer Scrooge, a grumpy old narrow man, gets visited by three ghosts who change the way he sees the world, in a good way."));

    KCmdLineArgs::init(argc, argv, &about);

    KCmdLineOptions options;
    options.add("in <file>", ki18nc("Application argument", "Input file"));
    options.add("out <file>", ki18nc("Application argument", "Output file"));
    options.add("param <name>", ki18nc("Application argument", "Name of a parameter"));
    options.add("value <value>", ki18nc("Application argument", "Value of a parameter"));

    options.add("", ki18nc("Application argument", "Input supported formats:"));
    options.add("", ki18n("%1").subs(SKGImportExportManager::getImportMimeTypeFilter()));
    options.add("", ki18nc("Application argument", "Output supported formats:"));
    options.add("", ki18n("%1").subs(SKGImportExportManager::getExportMimeTypeFilter()));
    KCmdLineArgs::addCmdLineOptions(options);

    QCoreApplication app(argc, argv);

    //Build list of arguments
    KCmdLineArgs* args = KCmdLineArgs::parsedArgs();
    SKGError err;
    if (!args->isSet("in")) err = SKGError(ERR_INVALIDARG, i18nc("Error message", "Missing -in option"));
    else if (!args->isSet("out")) err = SKGError(ERR_INVALIDARG, i18nc("Error message", "Missing -out option"));

    //Initialisation of a bank document
    SKGDocumentBank document;
    document.setComputeBalances(false);
    IFOKDO(err, document.initialize())
    IFOK(err) {
        //Import
        QString file = args->getOption("in");
        QFileInfo fi(file);
        if (fi.exists()) file = fi.absoluteFilePath();
        SKGImportExportManager imp(&document, KUrl::fromLocalFile(file));
        QMap<QString, QString>  parameters = imp.getImportParameters();
        parameters["import_by_load"] = 'Y';
        QStringList params = args->getOptionList("param");
        QStringList values = args->getOptionList("value");
        int nb = qMin(params.count(), values.count());
        for (int i = 0; i < nb; ++i) {
            parameters[params.at(i)] = values.at(i);
        }
        imp.setImportParameters(parameters);

        SKGTRACESEPARATOR;
        SKGTRACE << i18nc("Title of a console trace section", " Import parameters") << endl;
        QMapIterator<QString, QString> i(imp.getImportParameters());
        while (i.hasNext()) {
            i.next();
            SKGTRACE << i.key() << "=" << i.value() << endl;
        }
        SKGTRACESEPARATOR;
        SKGTRACE << i18nc("Title of a console trace section", " Imported file:") << imp.getFileName().prettyUrl()  << endl;
        if (fi.suffix().toUpper() == "SKG") {
            err = document.load(file, parameters["password"]);
        } else {
            SKGBEGINTRANSACTION(document, "IMPORT", err);
            IFOKDO(err, imp.importFile())
        }
    }
    IFOK(err) {
        //Export
        QString file = args->getOption("out");
        QFileInfo fi(file);
        if (fi.exists()) file = fi.absoluteFilePath();
        SKGImportExportManager exp(&document, KUrl::fromLocalFile(args->getOption("out")));
        SKGTRACESEPARATOR;
        SKGTRACE << i18nc("Title of a console trace section", " Exported file:") << exp.getFileName().prettyUrl()  << endl;
        if (fi.suffix().toUpper() == "SKG") {
            err = document.saveAs(file, true);
        } else {
            err = exp.exportFile();
        }
    }
    args->clear(); // Free up some memory.

    IFKO(err) {
        SKGTRACESUITE << err.getFullMessageWithHistorical() << endl;
        SKGTRACESEPARATOR;
        SKGTRACE << i18nc("Title of a console trace section", " FAILED") << endl;
        SKGTRACESEPARATOR;
    } else {
        SKGTRACESEPARATOR;
        SKGTRACE << i18nc("Title of a console trace section", " SUCCESSFUL") << endl;
        SKGTRACESEPARATOR;
    }

    SKGTraces::dumpProfilingStatistics();
    return err.getReturnCode();
}
