/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file implements classes SKGTrackerObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtrackerobject.h"
#include "skgsuboperationobject.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"

#include <klocale.h>

SKGTrackerObject::SKGTrackerObject(SKGDocument* iDocument, int iID): SKGNamedObject(iDocument, "v_refund", iID)
{
}

SKGTrackerObject::~SKGTrackerObject()
{
}

SKGTrackerObject::SKGTrackerObject(const SKGTrackerObject& iObject)
    : SKGNamedObject(iObject)
{
}

SKGTrackerObject::SKGTrackerObject(const SKGObjectBase& iObject)

{
    if (iObject.getRealTable() == "refund") {
        copyFrom(iObject);
    } else {
        *this = SKGNamedObject(iObject.getDocument(), "v_refund", iObject.getID());
    }
}

const SKGTrackerObject& SKGTrackerObject::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

SKGError SKGTrackerObject::createTracker(SKGDocumentBank* iDocument,
        const QString& iName,
        SKGTrackerObject& oTracker,
        bool iSendPopupMessageOnCreation)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGTrackerObject::createTracker", err);

    //Check if refund is already existing
    if (iName.isEmpty()) {
        oTracker = SKGTrackerObject(NULL, 0);
    } else if (iDocument) {
        iDocument->getObject("v_refund", "t_name='" % SKGServices::stringToSqlString(iName) % '\'', oTracker);
        if (oTracker.getID() == 0) {
            //No, we have to create it
            oTracker = SKGTrackerObject(iDocument);
            err = oTracker.setName(iName);
            if (!err) err = oTracker.save();

            if (!err && iSendPopupMessageOnCreation && iDocument) {
                err = iDocument->sendMessage(i18nc("Information message", "Tracker '%1' has been created" , iName), true);
            }
        }
    }

    return err;
}

SKGError SKGTrackerObject::getSubOperations(SKGListSKGObjectBase& oSubOperations) const
{
    SKGError err = getDocument()->getObjects("v_suboperation",
                   "r_refund_id=" % SKGServices::intToString(getID()) ,
                   oSubOperations);
    return err;
}

SKGError SKGTrackerObject::setClosed(bool iClosed)
{
    return setAttribute("t_close", iClosed ? "Y" : "N");
}

bool SKGTrackerObject::isClosed() const
{
    return (getAttribute("t_close") == "Y" ? true : false);
}

SKGError SKGTrackerObject::setComment(const QString& iComment)
{
    return setAttribute("t_comment", iComment);
}

QString SKGTrackerObject::getComment() const
{
    return getAttribute("t_comment");
}

double SKGTrackerObject::getCurrentAmount() const
{
    return SKGServices::stringToDouble(getAttribute("f_CURRENTAMOUNT"));
}

SKGError SKGTrackerObject::merge(const SKGTrackerObject& iTracker)
{
    SKGError err;

    SKGObjectBase::SKGListSKGObjectBase ops;
    if (!err) err = iTracker.getSubOperations(ops);
    int nb = ops.count();
    for (int i = 0; !err && i < nb; ++i) {
        SKGSubOperationObject op(ops.at(i));
        err = op.setTracker(*this);
        if (!err) err = op.save(true, false);
    }

    if (!err) err = iTracker.remove();
    return err;
}

#include "skgtrackerobject.moc"
