// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// specclass.cc
//
// Methods for working with the SpecClass class, determining for instance
//  temperature or diameter given the spectral class and magnitude

#include "specclass.h"

// For the SpecClass class -------------------------------------

SpecClass::SpecClass(const char * spectrum)
{
  if (!spectrum) {
    specstring = 0;
    major = 0; 
    minor = MKtype = 0.0;
    return;
  }
  specstring = new char[strlen(spectrum) + 1];
  strcpy(specstring, spectrum);

  if (isempty(specstring)) major = 0;
  else {
    unsigned int i = 0;
    while (specstring[i] && (isspace(specstring[i]) || specstring[i] == 'd'
                             || specstring[i] == 's'))
      i++;
    major = toupper(specstring[i]);
    if (major == 'R' || major == 'N' || major == 'C' || major == 'S')
      major = 'M';
  }

  minor = MKtype = 0.0;
}


// initialize(): Determine the spectral class subtype and MK luminosity class.
//  For reasons of speed, these are separated out into a separate function 
//  rather than being in the above constructor.

void SpecClass::initialize()
{
  stripspace(specstring);
  unsigned int i = 0, j;

  while (specstring[i] && !isdigit(specstring[i]))
    i++;
  j = i;
  while (specstring[j] && (isdigit(specstring[j]) || specstring[j] == '.'))
    j++;
  minor = (i != j) ? myatof(specstring + i) : 10;

  // luminosity class
  int size = strlen(specstring) - j + 1, posn = 0;
  char *temp = new char[size], *result = new char[size];
  bool flagsplit = false;
  strcpy(temp, specstring + j);

  // delete all extraneous characters
  for (unsigned int i = 0; i <= strlen(temp); i++) {
    if (temp[i] == 'I' || temp[i] == 'V' || !temp[i]) {
      result[posn++] = temp[i];
      continue;
    }
    if ((temp[i] == '/' || temp[i] == '-') && posn > 0) {
      flagsplit = true;
      result[posn] = 0;
      break;
    }
    if ((temp[i] == '+' || isupper(temp[i])) && posn > 0) {
      result[posn] = 0;
      break;
    }
  }

  // convert result from Roman Numeral to decimal
  if      (strcmp(result, "I")   == 0) MKtype = 1.0;
  else if (strcmp(result, "II")  == 0) MKtype = 2.0;
  else if (strcmp(result, "III") == 0) MKtype = 3.0;
  else if (strcmp(result, "IV")  == 0) MKtype = 4.0;
  else if (strcmp(result, "V")   == 0) MKtype = 5.0;
  else if (strcmp(result, "VI")  == 0) MKtype = 6.0;
  else MKtype = 0.0;

  if (flagsplit && MKtype >= 1.0) MKtype += 0.5;

  delete [] temp;
  delete [] result;
}


SpecClass::SpecClass(const SpecClass &sc)
{
  if (sc.specstring) {
    specstring = new char[strlen(sc.specstring) + 1];
    strcpy(specstring, sc.specstring);
  }
  else specstring = 0;

  major = sc.major;
  minor = sc.minor;
  MKtype = sc.MKtype;
}


SpecClass & SpecClass::operator = (const SpecClass &sc)
{
  if (this != &sc) {
    delete [] specstring;

    if (sc.specstring) {
      specstring = new char[strlen(sc.specstring) + 1];
      strcpy(specstring, sc.specstring);
    }
    else specstring = 0;

    major = sc.major;
    minor = sc.minor;
    MKtype = sc.MKtype;
  }
  return *this;
}


// temperature(): return the temperature of a star of this spectral class.
// These values are linear interpolations from the data presented in
//  Appendix E of _An Introduction to Modern Astrophysics_,
//  Carroll and Ostlie, 1996.

double SpecClass::temperature() const
{
  switch (major) {
    case 'W': return 50000;             // Kelvins, of course
    case 'O': return 59000 - minor * 2900;
    case 'B': return 30000 - minor * 2048;
    case 'A': return  9520 - minor *  232;
    case 'F': return  7200 - minor *  117;
    case 'G': return  6030 - minor *   78;
    case 'K': return  5250 - minor *  140;
    case 'M': return  3850 - minor *  151;
    default:  return     0;
  }
}


// diameter(): return the expected diameter, in light-years, of a star of
//  this spectral class with given visual magnitude.  We require bolometric
//  magnitude in order to get the correct diameter.  The bolometric
//  correction has been fit to this formula:
//
//  L  = log10(temp/Kelvins) - 4
//  BC = -8.499 L^4 + 13.421 L^3 - 8.131 L^2 - 3.901 L - 0.438
//
//  which was apparently published in the Journal of the RAS of Canada,
//  vol 92 no 1 pg 36.  I found it on this web page:
//  http://www.go.ednet.ns.ca/~larry/astro/HR_diag.html

double SpecClass::diameter(double magnitude) const
{
  double temp = temperature();
  if (temp <= 0.0) return 0;

  double logT = log10(temp) - 4;
  double bolcorr = -0.438 + logT * (-3.901 + logT * (-8.131 + logT *
				   (+13.421 - logT * 8.499)));

  // Now use the Stefan-Boltzmann law:
  //  R* = RSun * sqrt(L* / LSun) * (TSun / T*)^2
  //  but L* / LSun = 10^(0.4 * (MSun - M*))
  //  so R* = RSun * 10^(0.2 * (MSun - M*)) * (TSun / T*)^2
  //        = (const) / (T*)^2 * e^(-const * M*):

  return 46.146 * exp(-0.4605 * (magnitude + bolcorr)) / (temp * temp);
}


// Colors are defined in specclass.h

unsigned long SpecClass::color() const
{
  switch (major) {
    case 'W': return WOLF_RAYET_COLOR;
    case 'O': return O_COLOR;
    case 'B': return B_COLOR;
    case 'A': return A_COLOR;
    case 'F': return F_COLOR;
    case 'G': return G_COLOR;
    case 'K': return K_COLOR;
    case 'M': return M_COLOR;
    case 'D': return D_COLOR;
    case '*': return NON_STELLAR_COLOR;
    default:  return DEFAULT_COLOR;
  }
}

