# encoding: utf-8

require 'json'

module Github
  # Raised when GitHub returns any of the HTTP status codes
  module Error
    class ServiceError < GithubError
      attr_reader :http_headers, :body, :status

      MIN_BODY_LENGTH = 2

      def initialize(response)
        @http_headers = response[:response_headers]
        @body         = parse_body(response[:body])
        @status       = response[:status]

        super(create_message(response))
      end

      def create_message(response)
        "#{response[:method].to_s.upcase} #{response[:url]}: #{@status} #{@body}"
      end

      def decode_body(body)
        if body.respond_to?(:to_str) && body.length >= MIN_BODY_LENGTH
          JSON.parse(body, symbolize_names: true)
        else
          body
        end
      end

      def parse_body(body)
        body = decode_body(body)

        return '' if body.nil? || body.empty?

        if body[:error]
          body[:error]
        elsif body[:errors]
          error = Array(body[:errors]).first
          error.is_a?(Hash) ? error[:message] : error
        elsif body[:message]
          body[:message]
        else
          ''
        end
      end

      def self.http_status_code(code)
        define_method(:http_status_code) { code }
      end

      def self.errors
        @errors ||= Hash[
          descendants.map { |klass| [klass.new({}).http_status_code, klass] }
        ]
      end
    end # ServiceError
  end # Error
end # Github
