//rlp_math.cpp, Copyright (c) 2004-2006 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#define SWAP(a,b) {double temp=(a);(a)=(b);(b)=temp;}
#define _PREC 1.0e-12

extern Default defs;

static char *MRQ_error = 0L;
static double sqrt2pi = sqrt(_PI*2.0);

//---------------------------------------------------------------------------
//utilitity functions for memory allocation
double **dmatrix(int nrl, int nrh, int ncl, int nch)
{
	int i;
	double **m;

	m = (double **)malloc(nrh * sizeof(double*));
	//Allocate rows and set pointers to them
	for(i = 0; i < nrh; i++) {
		m[i] = (double *)malloc(nrh * sizeof(double));
		}
	return m;
}
void free_dmatrix(double **m, int nrl, int nrh, int ncl, int)
{
	int i;

	for(i = 0; i < nrh; i++) free(m[i]);
	free(m);
}

//---------------------------------------------------------------------------
//The routine gaussj solves linear equations by Gauss-Jordan elimination
bool gaussj(double **a, int n, double **b, int m)
{
	int *indxc, *indxr, *ipiv;
	int i, icol, irow, j, k, l, ll;
	double big, dum, pivinv;

	indxc = (int*)malloc(n*sizeof(int*));
	indxr = (int*)malloc(n*sizeof(int*));
	ipiv = (int*)malloc(n*sizeof(int*));
	for (j = 0; j < n; j++) ipiv[j] = 0;
	for (i = 0; i < n; i++) {				//This is the main loop over the
		big = 0.0;							//    columns to be reduced
		for(j = 0; j < n; j ++)				//This is the outer loop of the search
			if(ipiv[j] != 1)				//    for a pivot element
				for(k = 0; k < n; k ++) {
					if (ipiv[k] == 0) {
						if(fabs(a[j][k]) >= big) {
							big = fabs(a[j][k]);
							irow = j;				icol = k;
							}
						}
					else if(ipiv[k] > 1) {
						MRQ_error = "Singular Matrix (1)";
						free(ipiv);		free(indxr);	free(indxc);
						return false;
						}
				}
		++(ipiv[icol]);
		//We now have the pivot element, so we interchange rows, if needed,
		// to put the pivot element on the diagonal.
		if(irow != icol) {
			for(l = 0; l < n; l++) SWAP(a[irow][l], a[icol][l])
			for(l = 0; l < m; l++) SWAP(b[irow][l], b[icol][l])
			}
		indxr[i] = irow;		indxc[i] = icol;
		if(a[icol][icol] == 0.0) {
			MRQ_error = "Singular Matrix (2)";
			free(ipiv);		free(indxr);	free(indxc);
			return false;
			}
		pivinv = 1.0/a[icol][icol];
		a[icol][icol] = 1.0;
		for(l = 0; l < n; l++) a[icol][l] *= pivinv;
		for(l = 0; l < m; l++) b[icol][l] *= pivinv;
		for(ll = 0; ll <  n; ll++)
			if(ll != icol) { 							//Next, we reduce the rows
				dum = a[ll][icol];
				a[ll][icol] = 0.0;
				for(l = 0; l < n; l++) a[ll][l] -= a[icol][l]*dum;
				for(l = 0; l < m; l++) b[ll][l] -= b[icol][l]*dum;
				}
		}											// This is the end of the main loop
	for (l = n; l > 0; l--) {						//   over columns of the reduction.
		if(indxr[l] != indxc[l]) 					//   Unscramble the solution
			for(k = 0; k < n; k++) SWAP (a[k][indxr[l]], a[k][indxc[l]]);
		}											//And we are done.
	free(ipiv);		free(indxr);	free(indxc);
	return true;
}

//---------------------------------------------------------------------------
//The routine mrqcof is called by mrqmin to evaluate the linearized fitting
// matrix alpha and vector beta
void mrqcof(double x[], double y[], double z[], int ndata, double **a, int ma,
	int lista[], int mfit, double **alpha, double beta[], double *chisq,
	void (*funcs)(double, double, double **, double *, double *, int))
{
	int k, j, i;
	double ymod, wt, dy;
	double *dyda;

	dyda = (double*)malloc(ma*sizeof(double));
	for(j = 0; j < mfit; j++) {					//Initialize (symmetric) alpha, beta
		for(k = 0; k <= j; k++) alpha[j][k] = 0.0;
		beta[j] = 0.0;
		}
	*chisq = 0.0;
	for (i = 0; i < ndata; i++) {		 		//Summation loop over all data
		(*funcs)(x[i], z ? z[i] : 0.0, a, &ymod, dyda, ma);
		if(ymod != 0.0) dy = y[i]-ymod;			//functions = 0.0 if out of range
		else dy = 0.0;
		for(j = 0; j < mfit; j++) {
			wt = dyda[lista[j]];
			for (k = 0; k <= j; k++){
				alpha[j][k] += wt*dyda[lista[k]];
				}
			beta[j] += dy*wt;
			}
		(*chisq) += dy*dy; 							//And find X^2 if function o.k.
		}
	for(j = 0; j < mfit; j++)						//Fill the symmetric side
		for(k = 0; k <= j; k++) alpha[k][j]=alpha[j][k];
	free(dyda);
}

//---------------------------------------------------------------------------
//The routine mrqmin performs one iteration of Marquart's method for nonlinear
// parameter estimation
bool mrqmin(double *x, double *y, double *z, int ndata, double **a, int ma,
	int *lista, int mfit, double **covar, double **alpha, double *chisq,
	void (*funcs)(double, double, double **, double *, double *, int), double *alamda)
{
	int k, kk, j, ihit;
	static double *da, *atry, *beta, ochisq;
	static double **oneda, **atryref;

	if (*alamda < 0.0) {								//Initialization
		MRQ_error = 0L;
		oneda = dmatrix(1, mfit, 1, 1);
		atry = (double *)malloc(ma * sizeof(double));
		atryref = (double**)malloc(ma * sizeof(double*));
		for(j=0; j < ma; atryref[j++] = &atry[j]);
		da = (double*)malloc(ma *sizeof(double));
		beta = (double*)malloc(ma *sizeof(double));
		kk = mfit+1;
		for(j = 0; j < ma; j++) { 						//Does lista contain a proper
			ihit = 0;									//   permutation of the
			for(k = 0; k < mfit; k++)					//   coefficients ?
				if(lista[k] == j) ihit++;
			if(ihit == 0)
				lista[kk++] = j;
			else if (ihit >1) ErrorBox("Bad LISTA permutations in MRQMIN-1");
			}
		if(kk != ma+1) ErrorBox("Bad LISTA permutations in MRQMIN-2");
		*alamda = 0.001;
		mrqcof(x, y, z, ndata, a, ma, lista, mfit, alpha, beta, chisq, funcs);
		ochisq=(*chisq);
		}
	for (j = 0; j < mfit; j++) {						//Alter linearized fitting matrix
		for(k = 0; k < mfit; k++) covar[j][k] = alpha[j][k];	// by augmenting
		covar[j][j] = alpha[j][j]*(1.0+(*alamda));		// diagaonal elements
		oneda[j][0] = beta[j];
		}
	if (!gaussj(covar, mfit, oneda, 1)) return false;	//Matrix solution ?
	for(j = 0; j < mfit; j++) da[j] = oneda[j][0];
	if(*alamda == 0.0) {								//Once converged evaluate
														//  covariance matrix with
		free(beta);										//  alamda = 0.
		free(da);
		free(atry);
		free(atryref);
		free_dmatrix(oneda, 1, mfit, 1, 1);
		return true;
		}
	for(j = 0; j < ma; j++) atry[j] = *a[j];
	for(j = 0; j < mfit; j++)							//Did the trial succeed ?
		atry[lista[j]] = *a[lista[j]] + da[j];
	mrqcof(x, y, z, ndata, atryref, ma, lista, mfit, covar, da, chisq, funcs);
	if(*chisq < ochisq) {								//Success, accept the new solution
		*alamda *= 0.1;
		ochisq=(*chisq);
		for(j = 0; j < mfit; j++) {
			for(k = 0; k < mfit; k++) alpha[j][k] = covar[j][k];
			beta[j] = da[j];
			*a[lista[j]] = atry[lista[j]];
			}
		}
	else {												//Failure, increase almda and
		*alamda *= 10.0;								//    return.
		*chisq = ochisq;
		}
	return true;
}

bool Check_MRQerror()
{
	bool bRet;

	if(bRet = MRQ_error != 0L) ErrorBox(MRQ_error);
	MRQ_error = 0L;
	return bRet;
}

//---------------------------------------------------------------------------
//Use heap sort to sort elements of an float array
//W.H. pres, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1988/1989)
//Numerical Recipes in C, Cambridge University Press, ISBN 0-521-35465-X
// p. 245
void SortArray(int n, double *vals)
{
	int l, j, ir, i;
	double rra, *ra = vals-1;

	if(n < 2 || !vals) return;
	l=(n >> 1) + 1;				ir = n;
	for( ; ; ) {
		if(l > 1) rra = ra[--l];
		else {
			rra = ra[ir];		ra[ir] = ra[1];
			if(--ir == 1) {
				ra[1] = rra;	return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j] < ra[j+1]) ++j;
			if (rra < ra[j]) {
				ra[i] = ra[j];	j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i] = rra;
		}
}

//sorts array v1 making the corresponding rearrangement of v2
void SortArray2(int n, double *v1, double *v2)
{
	int l, j, ir, i;
	double rra, rrb, *ra = v1-1, *rb = v2-1;

	if(n < 2 || !v1 || !v2) return;
	l=(n >> 1) + 1;				ir = n;
	for( ; ; ) {
		if(l > 1) {
			rra = ra[--l];		rrb = rb[l];
			}
		else {
			rra = ra[ir];		rrb = rb[ir];
			ra[ir] = ra[1];		rb[ir] = rb[1];
			if(--ir == 1) {
				ra[1] = rra;	rb[1] = rrb;
				return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j] < ra[j+1]) ++j;
			if (rra < ra[j]) {
				ra[i] = ra[j];	rb[i] = rb[j];
				j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i] = rra;			rb[i] = rrb;
		}
}

//Use heap sort to sort elements of an xy array
void SortFpArray(int n, lfPOINT *vals)
{
	int l, j, ir, i;
	lfPOINT rra, *ra = vals-1;

	if(n < 2) return;
	l=(n >> 1) + 1;					ir = n;
	for( ; ; ) {
		if(l > 1) {
			rra.fx = ra[--l].fx; rra.fy = ra[l].fy;
			}
		else {
			rra.fx = ra[ir].fx;		rra.fy = ra[ir].fy;
			ra[ir].fx = ra[1].fx;	ra[ir].fy = ra[1].fy;	
			if(--ir == 1) {
				ra[1].fx = rra.fx;	ra[1].fy = rra.fy;
				return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j].fx < ra[j+1].fx) ++j;
			if (rra.fx < ra[j].fx) {
				ra[i].fx = ra[j].fx;	ra[i].fy = ra[j].fy;
				j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i].fx = rra.fx;				ra[i].fy = rra.fy;
		}
}

//---------------------------------------------------------------------------
// Cubic Spline Interpolation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 96 ff.
void spline(lfPOINT *v, int n, double *y2)
{
	int i, k;
	double p, qn, sig, un, *u;

	u = (double *)malloc(n * sizeof(double));
	y2[0] = u[0] = 0.0;
	for(i = 1; i < (n-1); i++) {
		sig = (v[i].fx-v[i-1].fx)/(v[i+1].fx-v[i-1].fx);
		p = sig*y2[i-1]+2.0;			y2[i]=(sig-1.0)/p;
		u[i]=(v[i+1].fy-v[i].fy)/(v[i+1].fx-v[i].fx)-(v[i].fy-v[i-1].fy)/(v[i].fx-v[i-1].fx);
		u[i]=(6.0*u[i]/(v[i+1].fx-v[i-1].fx)-sig*u[i-1])/p;
		}
	qn = un = 0.0;
	y2[n-1] = (un - qn * u[n-2])/(qn*y2[n-2]+1.0);
	for(k = n-2; k >= 0; k--) {
		y2[k] = y2[k]*y2[k+1]+u[k];
		}
	free(u);
}

//---------------------------------------------------------------------------
// The Gamma Function: return the ln(G(xx)) for xx > 0
// Ref: B.W. Brown, J. Lovato, K. Russel (1994)
//    DCDFLIB.C, Library of C Routinesfor Cumulative Distribution Functions,
//    Inverses, and other Parameters.

double devlpl(double a[], int n, double x)
{
	double term;
	int i;

	for(term = a[n-1], i= n-2; i>=0; i--) term = a[i] + term * x;
	return term;
}


double gammln(double x)
{
	static double coef[] = {0.83333333333333023564e-1,-0.27777777768818808e-2, 
	0.79365006754279e-3, -0.594997310889e-3, 0.8065880899e-3};
static double scoefd[] = {0.62003838007126989331e2, 0.9822521104713994894e1,
	-0.8906016659497461257e1, 0.1000000000000000000e1};
static double scoefn[] = {0.62003838007127258804e2, 0.36036772530024836321e2,
	0.20782472531792126786e2, 0.6338067999387272343e1,0.215994312846059073e1,
	0.3980671310203570498e0, 0.1093115956710439502e0,0.92381945590275995e-2,
	0.29737866448101651e-2};
	double offset, prod, xx;
	int i,n;

    if(x < 6.0) {
		prod = 1.0e0;	    xx = x;
		while(xx > 3.0) {
			xx -= 1.0;			prod *= xx;
			}
		if(x <= 2.0) while(xx < 2.0) {
			prod /= xx;			xx += 1.0;
			}
		// compute rational approximation to gamma(x)
		return log(devlpl(scoefn, 9, xx-2.0) / devlpl(scoefd, 4, xx-2.0) * prod);
		}
	else {
		offset = 0.91893853320467274178;	// hln2pi
		// if necessary make x at least 12 and carry correction in offset
		if(n = 13.0 >= x ? (int)(12.0 - x) : 0) xx = x;
		else {
			for(i=1, prod = 1.0; i<= n; i++) prod *= (x+(double)(i-1));
			offset -= log(prod);			xx = x+(double)n;
			}
		// compute power series
		return devlpl(coef, 5, 1.0/(xx*xx)) / xx + (offset+(xx-0.5)*log(xx)-xx);
		}
}

//---------------------------------------------------------------------------
// Special Functions
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 166 ff.

//The Factorial Function: return n!
double factrl(int n)
{
	static int ntop = 4;
	static double a[33]={1.0, 1.0, 2.0, 6.0, 24.0};
	int j;

	if(n < 0) return 0.0;		//error: no factorial for negative numbers
	if(n > 32) return exp(gammln(n+1.0));
	while(ntop < n) {			//fill in table up to desired value
		j = ntop++;		a[ntop]=a[j] * ntop;
		}
	return a[n];
}

//returns the incomplete gamma function evaluated by its series representation
void gser(double *gamser, double a, double x, double *gln)
{
	int n;
	double sum, del, ap;

	*gln = gammln(a);
	if(x <= 0) {
		*gamser = 0.0;			return;
		}
	else {
		ap = a;					del = sum = 1.0/a;
		for(n = 1; n <= 100; n++) {
			ap += 1.0;			del *= x/ap;		sum += del;
			if(fabs(del) <= fabs(sum) * _PREC) {
				*gamser = sum * exp(-x + a * log(x)-(*gln));
				return;
				}
			}
		// maximum number of iterations exceeded
		*gamser = sum * exp(-x + a * log(x)-(*gln));
		}

}

//returns the incomplete gamma function evaluated by its continued fraction representation
void gcf(double *gammcf, double a, double x, double *gln)
{
	int n;
	double gold=0.0, g, fac=1.0, b1=1.0, b0=0.0, anf, ana, an, a1, a0=1.0;

	*gln=gammln(a);		a1=x;
	for(n=1; n <= 100; n++) {
		an = (double)n;			ana = an -a;		a0 = (a1 + a0 * ana) * fac;
		b0 = (b1 + b0 * ana) *fac;					anf = an * fac;
		a1 = x * a0 + anf * a1;						b1 = x * b0 + anf * b1;
		if(a1) {
			fac = 1.0 / a1;							g = b1 * fac;
			if(fabs((g-gold)/g) <= _PREC) {
				*gammcf = exp(-x + a * log(x) -(*gln)) * g;
				return;
				}
			gold = g;
			}
		}
	// maximum number of iterations exceeded
	*gammcf = exp(-x + a * log(x) -(*gln)) * gold;
}

//returns the incomplete gamma function P(a,x)
double gammp(double a, double x)
{
	double gamser, gammcf, gln;

	if(x < 0.0 || a <= 0.0) return 0.0;
	if(x < (a+1.0)) {
		gser(&gamser, a, x, &gln);			return gamser;
		}
	else {
		gcf(&gammcf, a, x, &gln);			return 1.0-gammcf;
		}
	return 0.0;
}

//returns the complementary incomplete gamma function Q(a,x)
double gammq(double a, double x)
{
	double gamser, gammcf, gln;

	if(x < 0.0 || a <= 0.0) return 0.0;
	if(x < (a+1.0)) {
		gser(&gamser, a, x, &gln);			return 1.0-gamser;
		}
	else {
		gcf(&gammcf, a, x, &gln);			return gammcf;
		}
	return 0.0;
}

//continued fraction for incomplete beta function, used by betai()
double betacf(double a, double b, double x)
{
	double qap, qam, qab, em, tem, d, bz, bm = 1.0, bp, bpp, az = 1.0, am = 1.0, ap, app, aold;
	int m;

	qab = a+b;		qap = a+1.0;		qam = a-1.0;	bz = 1.0-qab*x/qap;
	for(m = 1; m <= 100; m++) {
		em = (double)m;			tem = em+em;
		d = em*(b-em)*x/((qam+tem)*(a+tem));
		ap = az + d * am;		bp = bz + d *bm;
		d = -(a+em)*(qab+em)*x/((qap+tem)*(a+tem));
		app = ap + d * az;		bpp = bp + d * bz;
		aold = az;				am = ap/bpp;
		bm = bp/bpp;			az = app/bpp;
		bz = 1.0;
		if(fabs(az-aold) <= (_PREC * fabs(az))) return az;	//success: return
		}
	return az;												//fail: iterations exceeded
}

//The incomplete beta function Ix(a,b) for 0 <= x <= 1
double betai(double a, double b, double x)
{
	double bt;

	if(x < 0.0 || x > 1.0) return 0.0;		//range !
	if(x == 0.0 || x == 1.0) bt = 0.0;
	else
		bt = exp(gammln(a+b)-gammln(a)-gammln(b)+a*log(x)+b*log(1.0-x));
	if(x < (a+1.0)/(a+b+2.0)) return bt * betacf(a, b, x)/a;
	else return 1.0 - bt * betacf(b, a, 1.0 - x)/b;
}

//The following relations are obviously based on:
//  Abramowitz, M. & Stegun I.A. (1964): Hanbook of Mathematical Functions.
//    Applied Mathematics Series, vol. 55 (Washington: National Bureau
//    of Standards).

//the binomial coefficient
double bincof(double n, double k)
{
	if(n<0 || k<0 || k > n) return 0.0;
	return exp(gammln(n+1.0) - gammln(k+1.0) - gammln(n-k+1.0));
}

//the cumulative binomial distribution
double binomdistf(double k, double n, double p)
{
	if(k > n || n < 0.0 || p < 0.0 || p >1.0) return 0.0;
	return betai(n-k, k+1, p);
}

//the beta function
double betaf(double z, double w)
{
	return exp(gammln(z)+gammln(w)-gammln(z+w));
}

//the error function: not all compilers have a built in erf()
double errf(double x)
{
	return x < 0.0 ? -gammp(0.5, x*x) : gammp(0.5, x*x);
}

//the complementary error function
double  errfc(double x)
{
//	return x < 0.0 ? 2.0 - gammq(0.5, x*x) : gammq(0.5, x*x);
	return x < 0.0 ? 1.0 + gammp(0.5, x*x) : gammq(0.5, x*x);
}

//cumulative normal distribution
double norm_dist(double x, double m, double s)
{
	return 0.5 + errf((x - m)/(s * _SQRT2))/2.0;
}

//normal distribution
double norm_freq(double x, double m, double s)
{
	double ex;

	ex = (x-m)/s;	ex = exp(-0.5*ex*ex);
	return ex/(s*sqrt2pi);
}

//cumulative exponential distribution
double exp_dist(double x, double l, double s)
{
	if(x >= 0.0 && l > 0.0) return 1.0-exp(-x*l);
	else return 0.0;
}

//inverse exponential distribution
double exp_inv(double p, double l, double s)
{
	if(p >= 1.0) return HUGE_VAL;
	if(l <= 0.0) return 0.0;
	return -log(1.0-p)/l;
}

//exponential distribution
double exp_freq(double x, double l, double s)
{
	if(x >= 0.0 && l > 0.0) return l*exp(-x*l);
	else return fabs(l);
}

//cumulative lognormal distribution
double lognorm_dist(double x, double m, double s)
{
	return 0.5 + errf((log(x) - m)/(s * _SQRT2))/2.0;
}

//lognormal distribution
double lognorm_freq(double x, double m, double s)
{
	double tmp;

	if(x > 0.0 && m > 0.0 && s > 0.0) {
		tmp = (log(x)-m)/s;
		return exp(-0.5*tmp*tmp)/(x*s*sqrt2pi);
		}
	return 0.0;
}

//chi square distribution
double chi_dist(double x, double df, double)
{
	return gammq(df/2.0, x/2);
}

//t-distribution
double t_dist(double t, double df, double)
{
	return betai(df/2.0, 0.5, (df/(df+t*t)));
}

//poisson distribution
double pois_dist(double x, double m, double)
{
	return gammq(x+1.0, m);
}

//f-distribution
double f_dist(double f, double df1, double df2)
{
	return betai(df2/2.0, df1/2.0, df2/(df2+df1*f));
}

//---------------------------------------------------------------------------
// Inverse of statitistical functions:
//    Use a combination of the Newton-Raphson method and bisection
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 273 ff.

// funcd supplies the function value fn and the derivative df of the function sf at x
void funcd(double x, double *fn, double *df, double (*sf)(double, double, double), 
		   double df1, double df2, double p)
{
	double y1, y2;

	*fn = (sf)(x, df1, df2);
	if(sf == norm_dist) *df = norm_freq(x, df1,df2);
	else {
		y1 = (sf)(x * 0.995, df1, df2);
		y2 = (sf)(x * 1.005, df1, df2);
		*df = (y2-y1)*100.0/x;
		}
	*fn = *fn - p;
}

//distinv does actual bisection and Newton-Raphson root finding
double distinv(double (*sf)(double, double, double), double df1, double df2, double p, double x0)
{
	int j = 0;
	double df, dx, dxold, f, fh, fl;
	double swap, temp, xh, xl, rts; 
	double x1 = x0/2.0, x2 = x0+x1;

	do {
		funcd(x1, &fl, &df, sf, df1, df2, p);
		funcd(x2, &fh, &df, sf, df1, df2, p);
		if(++j > 200) return 0.0;
		if(fl*fh < 0.0) break;
		dx = fabs(x2-x1);
		x1 = (x1 > dx) ? x1-dx : x1/2.0;			x2 += dx;
		}while(fl*fh >= 0.0);
	if(fl <= 0.0) {
		xl = x1;		xh = x2;
		}
	else {
		xh = x1;		xl = x2;
		swap = fl;		fl = fh;	fh = swap;
		}
	rts = x0;	dxold = fabs(x2-x1);	dx = dxold;
	funcd(rts, &f, &df, sf, df1, df2, p);
	for(j = 1; j <= 500; j++) {
        if((((rts-xh)*df-f)*((rts-xl)*df-f) >= 0.0) || (fabs(2.0*f) > fabs(dxold * df))) {
			dxold = dx;		dx = 0.5 * (xh-xl);		rts = xl + dx;
			if(xl == rts) return rts;
			}
		else {
			dxold = dx;		dx = f/df;		temp = rts;		rts -= dx;
			if(temp == rts) return rts;
			}
		if(fabs(dx) < _PREC) return rts;
		funcd(rts, &f, &df, sf, df1, df2, p);
		if(f < 0.0) {
			xl = rts;	fl = f;
			}
		else {
			xh = rts;	fh = f;
			}
		}
	return rts;
}

//---------------------------------------------------------------------------
//some statistical basics
//do quartiles, median of data
void d_quartile(int n, double *v, double *q1, double *q2, double *q3)
{
	int n2, n3;
	double f1, f2;

	if(!v || n<2) return;
	SortArray(n, v);			n2 = n >> 1;
	if(q1) {
		n3 = n2 >> 1;
		switch(n%4) {
		case 3:		n3 ++;		f1 = 2.0;		f2 = 2.0;		break;
		case 2:		n3 ++;		f1 = 3.0;		f2 = 1.0;		break;
		case 1:		n3 ++;		f1 = 4.0;		f2 = 0.0;		break;
		default:	f1 = 1.0;	f2 = 3.0;						break;
			}
		*q1 = (f1*v[n3-1] + f2*v[n3])/4.0;
		}
	if(q2) {
		if(n & 1) *q2 = v[n2];
		else *q2 = (v[n2-1] + v[n2])/2.0;
		}
	if(q3) {
		n3 = n2 >> 1;
		switch(n%4) {
		case 3:		n3++;		f1 = 2.0;		f2 = 2.0;	break;
		case 2:		f1 = 3.0;	f2 = 1.0;					break;
		case 1:		f1 = 4.0;	f2 = 0.0;					break;
		default:	f1 = 1.0;	f2 = 3.0;					break;
			}
		n3 += n2;
		*q3 = (f2*v[n3-1] + f1*v[n3])/4.0;
		}
}

// statistical basics partly based on
// Davies, J. and Gogh, B. (2000), GSL-1.7 - The GNU scientific library
//
//do variance
double d_variance(int n, double *v, double *mean, double *ss)
{
	int i;
	double d, m, va, e;

	for(i = 0, m = 0.0, d = 1.0; i < n; i++, d += 1.0) {
		m += (v[i] - m)/d;
		}
	if (mean) *mean = m;
	for(i = 0, va = 0.0, d = 1.0; i < n; i++, d += 1.0) {
		e = v[i] - m;		va += (e * e - va)/d;
		}
	if (ss) *ss = va * (double)n;
	return va * ((double)n/((double)(n-1)));
}

//do arithmethic mean
double d_amean(int n, double *v)
{
	int i;
	double d, mean;

	for(i = 0, mean = 0.0, d = 1.0; i < n; i++, d += 1.0) {
		mean += (v[i] - mean)/d;
		}
	return mean;
}


//do geometric mean
double d_gmean(int n, double *v)
{
	int i;
	double sum;

	for(i = 0, sum = 0.0; i < n; i++) {
		if(v[i] <= 0.0) return 0.0;
		sum += log(v[i]);
		}
	return exp(sum/n);
}

//do harmonic mean
double d_hmean(int n, double *v)
{
	int i;

	double sum;

	for(i = 0, sum = 0.0; i < n; i++) {
		if(v[i] == 0.0) return 0.0;
		sum += 1.0/(v[i]);
		}
	return (n/sum);
}

//kurtosis
double d_kurt(int n, double *v)
{
	double sum, avg, sd, tmp, dn = n;
	int i;

	for(i = 0, sum = 0.0; i < n; i++) sum += v[i];
	for(i = 0, avg = sum/dn, sum = 0.0; i < n; i++) sum += (tmp = v[i]-avg) * tmp;
	for(i = 0, sd = sqrt(sum/(dn-1.0)), sum=0.0; i < n; i++) sum += ((tmp = (v[i]-avg)/sd)*tmp*tmp*tmp);
	sum *= ((dn*(dn+1.0))/((dn-1.0)*(dn-2.0)*(dn-3.0)));
	tmp = (3.0 * (dn-1.0) * (dn-1.0))/((dn-2.0)*(dn-3.0));
	return sum - tmp;
}

//skewness
double d_skew(int n, double *v)
{
	double sum, avg, sd, tmp, dn = n;
	int i;

	for(i = 0, sum = 0.0; i < n; i++) sum += v[i];
	for(i = 0, avg = sum/dn, sum = 0.0; i < n; i++) sum += (tmp = v[i]-avg) * tmp;
	for(i = 0, sd = sqrt(sum/(dn-1.0)), sum=0.0; i < n; i++) sum += ((tmp = (v[i]-avg)/sd)*tmp*tmp);
	return sum * dn/((dn-1.0)*(dn-2.0));
}

//---------------------------------------------------------------------------
// Create a frequency distribution by counting the elements which may be 
// assigned to a bin
double d_classes(DataObj *d, double start, double step, double *v, int nv, char *range)
{
	int i, j, r, c, nc, *f;
	AccRange *ar;

	if(!range || !nv || !v || step <= 0.0 || !(ar = new AccRange(range))) return 0.0;
	if(!(nc = ar->CountItems()) || !ar->GetFirst(&c, &r) || !(f=(int*)calloc(nc, sizeof(int)))) {
		delete ar;				return 0.0;
		}
	for(i = 0; i < nv; i++) {
		j = (int)(floor((v[i] - start)/step));
		if(j < 0) j = 0;		if(j >= nc) j = (nc-1);
		f[j]++;
		}
	for( ; nc > 0 && !(f[nc-1]); nc--);
	for(i = 0; ar->GetNext(&c, &r) && i < nc; i++) {
		d->SetValue(r, c, (double)f[i]);
		}
	free(f);					return ((double)nv);
}

//---------------------------------------------------------------------------
// Pearsons linear correlation
// (1) W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 503 ff.
// (2) B. Gough (2000), linear.c, gsl-1.7 the GNU scientific library
double d_pearson(double *x, double *y, int n, char *dest, DataObj *data)
{
	int j, r, c;
	double yt, xt, t, df, res[4];
	double syy=0.0, sxy=0.0, sxx=0.0, ay=0.0, ax=0.0;
	AccRange *rD;

	for(j = 0;	j < n; j++) {				// find means
		ax += (x[j] - ax) / (j+1);			ay += (y[j] - ay) / (j+1);
		}
	for(j = 0; j < n; j++) {				// correlation
		xt = x[j] - ax;						yt = y[j] - ay;
		sxx += (xt*xt-sxx) / (j+1);			syy += (yt*yt-syy) / (j+1);
		sxy += (xt*yt-sxy) / (j+1);
		}
	res[0] = sxy/sqrt(sxx*syy);				//pearsons r
	if(dest) {
		res[1] = 0.5 * log((1.0+res[0]+_PREC)/(1.0-res[0]+_PREC));	//Fishers z-transform
		df = n-2;
		t = res[0]*sqrt(df/((1.0-res[0]+_PREC)*(1.0+res[0]+_PREC)));	//Student's t
		res[2] = betai(0.5*df, 0.5, df/(df+t*t));					//probability
		res[3] = n;
		}
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 4 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return res[0];
}

//---------------------------------------------------------------------------
// Given an array w, rank returns the rank of v1 in v
// if v1 is not found in v 0 is returned
double d_rank(int n, double *v, double v1)
{
	double *sv;
	int i, j;

	if(!n || !v) return 0.0;		if(n < 2) return 1.0;
	if(!(sv = (double*)memdup(v, n * sizeof(double), 0))) return 0.0;
	SortArray(n, sv);
	for(i = j = 0; i < n; i++) {
		if(v1 == sv[i]) {
			for( ;(i+j)<n; j++) if(sv[i+j] > v1) break;
			free(sv);				return (double)i + 1.0 + (((double)j-1.0)/2.0);
			}
		}
	free(sv);						return 0.0;
}

//---------------------------------------------------------------------------
// Spearman rank-order correlation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Recipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 507 ff.

//Given a sorted array w, crank replaces the elements by their rank
void crank(int n, double *w0, double *s)
{
	int j=1, ji, jt;
	double t, rank, *w = w0-1;

	*s = 0.0;
	while (j < n) {
		if(w[j+1] != w[j]) {
			w[j] = j;		++j;
			}
		else {
			for(jt = j+1; jt <= n; jt++)
				if(w[jt] != w[j]) break;
			rank = 0.5 * (j+jt-1);
			for(ji = j; ji <= (jt-1); ji++) w[ji] = rank;
			t = jt -j;
			*s += t*t*t -t;
			j = jt;
			}
		}
	if(j == n) w[n] = n;
}

//the actual rank correlation
double d_spearman(double *x, double *y, int n, char *dest, DataObj *data)
{
	int j, r, c;
	double vard, t, sg, sf, fac, en3n, en, df, aved, tmp;
	double res[6];
	AccRange *rD;

	SortArray2(n, x, y);		crank(n, x, &sf);
	SortArray2(n, y, x);		crank(n, y, &sg);
	for(j = 0, res[0] = 0.0; j < n; j++) res[0] += ((tmp = (x[j]-y[j]))*tmp);
	en = n;						en3n = en*en*en -en;
	aved = en3n/6.0 - (sf+sg)/12.0;
	fac = (1.0-sf/en3n)*(1.0-sg/en3n);
	vard = ((en-1.0)*en*en*((en+1.0)*(en+1.0))/36.0)*fac;
	vard = ((en-1.0)*en*en*((tmp = (en+1.0))*tmp)/36.0)*fac;
	res[1] = (res[0]-aved)/sqrt(vard);
	res[2] = errfc(fabs(res[1])/_SQRT2);
	res[3] = (1.0-(6.0/en3n)*(res[0]+0.5*(sf+sg)))/fac;
	t = res[3]*sqrt((en-2.0)/((res[3]+1.0)*(1.0-res[3])));
	df = en-2.0;
    res[4] = betai(0.5*df, 0.5, df/(df+t*t));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);	res[5] = n;
		for(j = 0; j < 6 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return res[3];
}

//---------------------------------------------------------------------------
// Kendal's non-parametric correlation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Recipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 510 ff.

double d_kendall(double *x, double *y, int n, char *dest, DataObj *data)
{
	int j, k, n1, n2, is, r, c;
	double aa, a1, a2, sv, res[4];
	AccRange *rD;

	for (j = n1 = n2 = is = 0; j < (n-1); j++) {
		for(k = j+1; k < n; k++) {
			a1 = x[j] - x[k];		a2 = y[j] - y[k];		aa = a1*a2;
			if(aa != 0.0) {
				n1++;				n2++;
				if (aa > 0.0) is++;
				else is--;
				}
			else {
				if(a1 != 0.0) n1++;	if(a2 != 0.0) n2++;
				}
			}
		}
	res[0] = ((double)is)/(sqrt((double)n1) * sqrt((double)n2));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		sv = (4.0 * ((double)n) + 10.0)/(9.0*((double)n)*((double)(n-1)));
		res[1] = res[0]/sqrt(sv);	res[2] = errfc(fabs(res[1])/_SQRT2);
		res[3] = n;			rD->GetFirst(&c, &r);
		for(j = 0; j < 4 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return res[0];
}


//linear regression
double d_regression(double *x, double *y, int n, char *dest, DataObj *data)
{
	double sx, sy, dx, dy, sxy, sxx, syy, sdy, df;
	double res[10];		// slope, intercept, mean x, mean y, SE of slope, 
						//   variance(x), variance(y), variance(fit), F of regression, significance
	int i, j, r, c;
	AccRange *rD;

	if(n < 2) return 0.0;
	for(i = 0, 	sx = sy = 0.0; i < n; i++) {
		sx += x[i];			sy += y[i];
		}
	res[2] = sx /n;			res[3] = sy/n;
	sxy = sxx = syy = 0.0;
	for(i = 0; i < n; i++) {
		dx = x[i]-res[2];	dy = y[i]-res[3];
		sxx += (dx*dx);		syy += (dy*dy);		sxy += (dx*dy);
		}
	res[0] = sxy / sxx;		res[1] = res[3] - res[0] * res[2];
	for(i = 0, sdy = 0.0; i < n; i++) {
		dy = y[i] - (res[1] + x[i] *res[0]);
		sdy += (dy * dy);
		}
	sdy = sdy/(n-2);		res[4] = sqrt(sdy/sxx);		df = (n-2);
	res[5] = sxx/(n-1);		res[6] = syy/(n-1);			res[7] = sdy;
	res[8] = sxy/sdy*sxy/sxx;
	res[9] = betai(df/2.0, 0.5, df/(df+res[8]));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 10 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return n;
}

//covariance
double d_covar(double *x, double *y, int n, char *dest, DataObj *data)
{
	int i;
	double sx, sy, dx, dy, sxy;

	if(n < 2) return 0.0;
	for(i = 0, 	sx = sy = 0.0; i < n; i++) {
		sx += x[i];			sy += y[i];
		}
	sx /= n;		sy /= n;		sxy = 0.0;
	for(i = 0; i < n; i++) {
		dx = x[i]-sx;		dy = y[i]-sy;
		sxy += (dx*dy - sxy) / (i+1);
		}
	return sxy;
}

//Mann-Whitney U Test
double d_utest(double *x, double *y, int n1, int n2, char *dest, DataObj *data)
{
	double *da, *ta, u1, u2, su, tmp;
	double res[7];
	AccRange *rD;
	int i, j, n, r, c;

	if(!x || !y || n1 < 2 || n2 < 2) return 0.0;
	da = (double*)malloc((n = (n1+n2)) * sizeof(double));
	ta = (double*)malloc(n * sizeof(double));
	if(!da || !ta) {
		if(da) free(da);	if(ta) free(ta); return 0.0;
		}
	for(i = 0; i < n1; i++) {
		da[i] = x[i];		ta[i] = 1.0;
		}
	for(j = 0; j < n2; j++) {
		da[i] = y[j];		ta[i++] = 2.0;
		}
	SortArray2(n, da, ta);	crank(n, da, &tmp);
	for(i = 0, res[0] = res[1] = 0.0; i < n; i++) {
		if(ta[i] == 1.0) res[0] += da[i];
		else res[1] += da[i];
		}
	free(da);									free(ta);
	u1 = (n1*n2 + (n1*(n1+1))/2.0) - res[0];	u2 = (n1*n2 + ((n2+1)*n2)/2.0) - res[1];
	su = sqrt((n1*n2*(n1+n2+1))/12.0);			res[2] = u2 > u1 ? u2 : u1;
	res[3] = (res[2] - (n1*n2)/2.0)/su;			res[6] = errfc(res[3]/_SQRT2);
	if((dest) && (data) && (rD = new AccRange(dest))) {
		res[4] = n1;							res[5] = n2;
		rD->GetFirst(&c, &r);
		for(i = 0; i < 7 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return res[6];
}

//t-test
double d_ttest(double *x, double *y, int n1, int n2, char *dest, DataObj *data)
{
	int i, r, c;
	double sx, sy, mx, my, d, df, p;
	double res[9];			// mean1, SD1, n1, mean2, SD2, n2, p if variances equal,
	AccRange *rD;			//    corrected df, corrected p

	d_variance(n1, x, &mx, &sx);		d_variance(n2, y, &my, &sy);
	d = ((sx+sy)/(n1+n2-2)) * ((double)(n1+n2)/(double)(n1*n2));
	d = (mx-my)/sqrt(d);	//Student's t

	//Welch's correction for differences in variance
	df = (sx/(double)n1)*(sx/(double)n1)/(double)(n1+1)+(sy/(double)n2)*(sy/(double)n2)/(double)(n2+1);
	df = (sx/(double)n1+sy/(double)n2)*(sx/(double)n1+sy/(double)n2)/df;
	df -= 2.0;

//	an alternative formula for correction
//	p = (sx/(double)n1)*(sx/(double)n1)/(double)(n1-1) + (sy/(double)n2)*(sy/(double)n2)/(double)(n2-1);
//	df = (sx/(double)n1 + sy/(double)n2) * (sx/(double)n1 + sy/(double)n2) / p;

	p = betai(df/2.0, 0.5, (df/(df+d*d)));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		res[0] = mx;	res[1] = sqrt(sx/(double)(n1-1));	res[2] = n1;
		res[3] = my;	res[4] = sqrt(sy/(double)(n2-1));	res[5] = n2;
		res[7] = df;	df = (n1-1) + (n2-1);	res[6] = betai(df/2.0, 0.5, (df/(df+d*d)));
		res[8] = p;
		rD->GetFirst(&c, &r);
		for(i = 0; i < 9 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return p;
}

//t-test for paired samples
double d_ttest2(double *x, double *y, int n, char *dest, DataObj *data)
{
	double sx, sy, mx, my, df, cov, sd, t, p;
	int i, r, c;
	double res[6];			// mean1, SD1, mean2, SD2, n, p 
	AccRange *rD;

	d_variance(n, x, &mx, &sx);		d_variance(n, y, &my, &sy);
	sx = d_variance(n, x, &mx);		sy = d_variance(n, y, &my);
	cov = d_covar(x, y, n, 0L, 0L) * ((double)n/(double)(n-1));
	sd = sqrt((sx+sy-2*cov)/n);
	t = (mx-my)/sd;					df = (n-1);
	p = betai(0.5*df, 0.5, df/(df+t*t));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		res[0] = mx;	res[1] = sqrt(sx);	res[5] = p;
		res[2] = my;	res[3] = sqrt(sy);	res[4] = n;
		rD->GetFirst(&c, &r);
		for(i = 0; i < 6 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return p;
}

//f-test
double d_ftest(double *x, double *y, int n1, int n2, char *dest, DataObj *data)
{
	int i, r, c;
	double sx, sy, mx, my, d, df1, df2, p;
	double res[6];			// mean1, SD1, n1, mean2, SD2, n2
	AccRange *rD;

	for(i=0, sx = 0.0; i < n1; sx += x[i], i++);				mx = sx/n1;
	for(i=0, sy = 0.0; i < n2; sy += y[i], i++);				my = sy/n2;
	for(i=0, sx = 0.0; i < n1; sx += ((d=(x[i]-mx))*d), i++);	sx /= (n1-1);
	for(i=0, sy = 0.0; i < n2; sy += ((d=(y[i]-my))*d), i++);	sy /= (n2-1);
	if(sx > sy) {
		d = sx/sy;		df1 = n1-1;		df2 = n2-1;
		}
	else {
		d = sy/sx;		df1 = n2-1;		df2 = n1-1;
		}
	p = 2.0 * betai(df2/2.0, df1/2.0, df2/(df2+df1*d));
	if(p > 1.0) p = 2.0-p;
	if((dest) && (data) && (rD = new AccRange(dest))) {
		res[0] = mx;	res[1] = sqrt(sx);	res[2] = n1;
		res[3] = my;	res[4] = sqrt(sy);	res[5] = n2;
		rD->GetFirst(&c, &r);
		for(i = 0; i < 6 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return p;
}

//---------------------------------------------------------------------------
// Calendar, Date- and time functions
// The following characters are used as format specifiers in a format string,
//    all other characters are either ignored or copyied to the output
//
//    Y   four digits year               y    two digits year
//    X   month's full name              x    three character month name
//    Z   two digits day of month        z    same as Z but no leading zero
//    V   two digit month number         v    number of month
//    W   single letter month
//    D   full name of day               d    three characters for day name
//    E   two digits weekday             e    one or two digits weekday
//    F   single character day name
//    H   two digits for hours           h    hours with no leading zero
//    M   two digits for minutes         m    minutes with no leading zero
//    S   two digits for seconds         s    seconds with no leading zero
//    T   two digits seconds, two dec.   t    same as T but no leading zero
//    U   full precision seconds

static char *dt_month[] = {"January", "February", "March", "April", "May", "June",
	"July", "August", "September", "October", "November", "December"};

static char *dt_months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug",
	"Sep", "Oct", "Nov", "Dec"};

static int dt_monthl[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

static char *dt_day[] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday",
	"Friday", "Saturday"};

static char *dt_days[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};

typedef struct rlp_datetime {
	int aday, year, doy, month, dom, dow, hours, minutes;
	double seconds;
}rlp_datetime;

static bool leapyear(int year) {
	return ((year % 4 == 0 && year % 100 != 0) || year % 400 == 0);
}

static int year2aday(int y)
{
	int aday, y1;

	y1 = y - 1900;
	aday = y1 * 365;
	aday += ((y1-1) >> 2 );
	aday -= (y1 / 100);
	aday += ((y/400)-4);
	return aday;
}

static void set_dow(rlp_datetime *dt)
{
	dt->dow = (dt->aday %7)+1;
}

static int parse_date (rlp_datetime *dt, char *src, char *fmt)
{
	int i, j, k;
	char tmp_str[10];

	if(!src || !src[0] || !fmt || !fmt[0]) return 0;
	if(*src == '\'') src++;
	for(i = j = 0; fmt[i] && src[j]; i++) {
		switch (fmt[i]) {
		case 'Y':		case 'y':			// year is numeric
			if(j && src[j] == '-' || src[j] == '/' || src[j] == '.') j++;
			if(sscanf(src+j, "%d", &dt->year)) {
				if(dt->year < 0) return 0;
				while(isdigit(src[j])) j++;
				if(dt->year<60) dt->year += 2000;
				else if(dt->year <99) dt->year += 1900;
				}
			else return 0;
			break;
		case 'X':		case 'x':			// month can be text
			if(j && src[j] == '-' || src[j] == '/' || src[j] == '.') j++;
			tmp_str[0] = toupper(src[j]);
			tmp_str[1] = tolower(src[j+1]);
			tmp_str[2] = tolower(src[j+2]);
			tmp_str[3] = 0;
			for(k = dt->month = 0; k < 12; k++) {
				if(0 == strcmp(tmp_str,dt_months[k])) {
					dt->month = k+1;			break;
					}
				}
			if(dt->month) while(isalpha(src[j])) j++;
			else return 0;
			break;
		case 'V':		case 'v':			//    or numeric
			if(j && src[j] == '-' || src[j] == '/' || src[j] == '.') j++;
			if(sscanf(src+j, "%d", &dt->month)) {
				if(dt->month <= 0 || dt->month > 12) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'Z':		case 'z':			// day of month is numeric
			if(j && src[j] == '-' || src[j] == '/' || src[j] == '.') j++;
			if(sscanf(src+j, "%d", &dt->dom)) {
				if(dt->dom <= 0 || dt->dom > 31) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'H':		case 'h':			// hours are numeric
			if(sscanf(src+j, "%2d", &dt->hours)) {
				if(dt->hours < 0 || dt->hours > 23) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'M':		case 'm':			// minutes are numeric
			if(j && src[j] == ' ' || src[j] == ':') j++;
			if(sscanf(src+j, "%2d", &dt->minutes)) {
				if(dt->minutes < 0 || dt->minutes >= 60) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'S':		case 's':			// seconds are numeric
		case 'T':		case 't':
			if(j && src[j] == ' ' || src[j] == ':') j++;
			if(sscanf(src+j, "%lf", &dt->seconds)) {
				if(dt->seconds < 0.0 || dt->seconds >= 60.0) return 0;
				while(isdigit(src[j]) || src[j] == '.') j++;
				}
			else return 0;
			dt->seconds += 1.0e-12;
			break;
		default:
			if(fmt[i] && fmt[i] == src[j]) j++;
			}
		}
	if(dt->year && dt->month && dt->dom) {
		for(dt->doy = 0, i = dt->month-2; i >= 0; i--) {
			if(i == 1) dt->doy += leapyear(dt->year) ? 29 : 28;
			else dt->doy += dt_monthl[i]; 
			}
		dt->doy += dt->dom;
		if(dt->year >= 1900) dt->aday = year2aday(dt->year);
		dt->aday += dt->doy;
		}
	return j;
}

static char *date2text(rlp_datetime *dt, char *fmt)
{
	static char res[80];
	int i, pos;
	double secs;

	res[0] = 0;
	if(!fmt || !fmt[0] || !dt) return res;
	set_dow(dt);
	secs = dt->seconds;
	if (secs > 59.4999) secs = 59.4999;
	for(pos = i = 0; fmt[i] && pos < 70; i++) {
		switch(fmt[i]) {
		case 'Y':
			if(dt->year) pos+=sprintf(res+pos, "%4d", dt->year);
			else pos += sprintf(res+pos, "####");		break;
		case 'y':
			if(dt->year) pos+=sprintf(res+pos, "%02d", (dt->year %100));
			else pos += sprintf(res+pos, "##");			break;
		case 'Z':
			if(dt->dom) pos+=sprintf(res+pos, "%02d", dt->dom);
			else pos += sprintf(res+pos, "##");			break;
		case 'z':
			if(dt->dom) pos+=sprintf(res+pos, "%d", dt->dom);
			else pos += sprintf(res+pos, "##");			break;
		case 'X':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%s", dt_month[dt->month-1]);
			else pos += sprintf(res+pos, "###");		break;
		case 'x':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%s", dt_months[dt->month-1]);
			else pos += sprintf(res+pos, "###");		break;
		case 'V':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%02d", dt->month);
			else pos += sprintf(res+pos, "##");			break;
		case 'v':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%d", dt->month);
			else pos += sprintf(res+pos, "##");			break;
		case 'W':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%c", dt_month[dt->month-1][0]);
			else pos += sprintf(res+pos, "#");			break;
		case 'D':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%s", dt_day[dt->dow-1]);
			else pos += sprintf(res+pos, "###");		break;
		case 'd':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%s", dt_days[dt->dow-1]);
			else pos += sprintf(res+pos, "###");		break;
		case 'E':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%02d", dt->dow);
			else pos += sprintf(res+pos, "##");			break;
		case 'e':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%d", dt->dow);
			else pos += sprintf(res+pos, "##");			break;
		case 'F':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%c", dt_day[dt->dow-1][0]);
			else pos += sprintf(res+pos, "#");			break;
		case 'H':
			if(dt->hours >=0 && dt->hours < 24) pos+=sprintf(res+pos, "%02d", dt->hours);
			else pos += sprintf(res+pos, "##");			break;
		case 'h':
			if(dt->hours >=0 && dt->hours < 24) pos+=sprintf(res+pos, "%d", dt->hours);
			else pos += sprintf(res+pos, "##");			break;
		case 'M':
			if(dt->minutes >=0 && dt->minutes < 60) pos+=sprintf(res+pos, "%02d", dt->minutes);
			else pos += sprintf(res+pos, "##");			break;
		case 'm':
			if(dt->minutes >=0 && dt->minutes < 60) pos+=sprintf(res+pos, "%d", dt->minutes);
			else pos += sprintf(res+pos, "##");			break;
		case 'S':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%02d", iround(secs));
			else pos += sprintf(res+pos, "##");			break;
		case 's':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%d", iround(secs));
			else pos += sprintf(res+pos, "##");			break;
		case 'T':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%02.2lf", dt->seconds);
			else pos += sprintf(res+pos, "##.##");		break;
		case 't':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%.2lf", dt->seconds);
			else pos += sprintf(res+pos, "##.##");		break;
		default:
			pos += sprintf(res+pos, "%c", fmt[i]);		break;
			}
		}
	res[pos] = 0;
	return res;
}

static double date2value(rlp_datetime *dt)
{
	double res;

	if(!dt) return 0.0;

	res = dt->seconds/60.0 + (double)dt->minutes;
	res = res/60.0 + (double)dt->hours;
	res = res/24.0 + (double)dt->aday;
	return res;
}

static void parse_datevalue(rlp_datetime *dt, double dv)
{
	int i, j, d;

	if(!dt || dv < 0.0) return;
	if(dv > 1.0) {
		dt->aday = (int)floor(dv);
		dt->year = (int)(dv/365.2425);
		d = (int)floor(dv);
		do {
			dt->doy = d - 365*dt->year;
			dt->doy -= ((dt->year-1)>>2);
			dt->doy += ((dt->year)/100);
			dt->doy -= ((dt->year+300)/400);
			if(dt->doy < 1) dt->year--;
			}while(dt->doy < 1);
		dt->year += 1900;
		for(i = dt->month = 0, d = dt->doy; i < 12 && d > 0; i++) {
			if(i == 1 && d > (j = (leapyear(dt->year)) ? 29 : 28)) d -= j;
			else if(i != 1 && d > dt_monthl[i]) d -= dt_monthl[i];
			else break;
			}
		dt->month = i+1;				dt->dom = d;
		}
	dv -= floor(dv);				dv *= 24.0;
	dt->hours = (int)floor(dv);		dv -= floor(dv);
	dv *= 60.0;						dt->minutes = (int)floor(dv); 
	dv -= floor(dv);				dt->seconds = dv *60.0 + 1.0e-12; 
	if(dt->seconds > 59.9999) {
		dt->seconds = 0.0;			dt->minutes++;
		if(dt->minutes == 60) {
			dt->hours++;			dt->minutes = 0;
			}
		}
}

static char *dt_popfmt[] = {"Z.V.Y H:M:S", "Z/V/Y H:M:S", "Z-V-Y H:M:S", "Z.X.Y H:M:S",
	"Y.V.Z H:M:S", "Y-X-Z H:M:S", "H:M:S", 0L};

bool date_value(char *desc, char *fmt, double *value)
{
	int i;
	rlp_datetime dt;

	dt.year = dt.aday = dt.doy = dt.month = dt.dom = dt.dow = dt.hours = dt.minutes = 0;
	dt.seconds = 0.0;
	if(!value || !desc || !desc[0]) return false;
	if(fmt && fmt[0]) {
		if(parse_date(&dt, desc, fmt)) {
			*value = date2value(&dt);	return true;
			}
		}
	else {
		if(parse_date(&dt, desc, defs.fmt_datetime)) {
			*value = date2value(&dt);	return true;
			}
		}
	for(i=0; dt_popfmt[i]; i++) {
		if(parse_date(&dt, desc, dt_popfmt[i])) {
			*value = date2value(&dt);	return true;
			}
		}
	return false;
}

char *value_date(double dv, char *fmt)
{
	rlp_datetime dt;

	parse_datevalue(&dt, dv);
	return date2text(&dt, fmt ? fmt : defs.fmt_date);
}

double now_today()
{
	double res = 0.0;
	time_t ti = time(0L);

	date_value(ctime(&ti)+4, "x z H:M:S Y", &res);
	return res;
}

void split_date(double dv, int *y, int *mo, int *dom, int *dow, int *doy, int *h, int *m, double *s)
{
	rlp_datetime dt;

	parse_datevalue(&dt, dv);
	set_dow(&dt);
	if(y) *y = dt.year;				if(mo) *mo = dt.month;
	if(dom) *dom = dt.dom;			if(dow) *dow = dt.dow;
	if(doy) *doy = dt.doy;			if(h) *h = dt.hours;
	if(m) *m = dt.minutes;			if(s) *s = dt.seconds;
}

//---------------------------------------------------------------------------
// Use the Delauney triangulation to create a 3D mesh of dispersed data
//
class Triangle {
public:
	Triangle *next;
	fPOINT3D pt[4];

	void SetRect();
	bool TestVertex(double x, double y);

private:
	double cx, cy, r2;				//circumcircle
	fRECT rc;						//bounding rectangle
	lfPOINT ld[3];					//line eqations
};

class Triangulate {
public:
	Triangle *trl;

	Triangulate(Triangle *t_list);
	bool AddEdge(fPOINT3D *p1, fPOINT3D *p2);
	bool AddVertex(fPOINT3D *v);

private:
	typedef struct edge {
		edge *next;
		fPOINT3D p1, p2;
	};
	edge *edges;
};

void
Triangle::SetRect()
{
	int i, i2;
	double dy1, dy2, dx, dy;
	double m1, m2, mx1, mx2, my1, my2;

	//setup bounding rectangle
	rc.Xmin = rc.Xmax = pt[0].fx;	rc.Ymin = rc.Ymax = pt[0].fy;
	for(i = 1; i < 3; i++) {
		if(pt[i].fx < rc.Xmin) rc.Xmin = pt[i].fx;
		if(pt[i].fx > rc.Xmax) rc.Xmax = pt[i].fx;
		if(pt[i].fy < rc.Ymin) rc.Ymin = pt[i].fy;
		if(pt[i].fy > rc.Ymax) rc.Ymax = pt[i].fy;
		}
	//get three line equations in 2D
	for(i = 0; i < 3; i++) {
		i2 = (i+1)%3;
		ld[i].fx = pt[i].fy;
		if(pt[i].fx != pt[i2].fx) {
			ld[i].fy = (pt[i2].fy - pt[i].fy) / (pt[i2].fx - pt[i].fx);
			}
		else ld[i].fy = HUGE_VAL;
		}
	//close polygon
	pt[3].fx = pt[0].fx;	pt[3].fy = pt[0].fy;	pt[3].fz = pt[0].fz;
	//circumcricle
	dy1 = fabs(pt[0].fy - pt[1].fy);			dy2 = fabs(pt[1].fy - pt[2].fy);
	m1 = (pt[0].fx - pt[1].fx)/(pt[1].fy - pt[0].fy);
	m2 = (pt[1].fx - pt[2].fx)/(pt[2].fy - pt[1].fy);
	mx1 = (pt[0].fx + pt[1].fx)/2.0;			my1 = (pt[0].fy + pt[1].fy)/2.0;
	mx2 = (pt[1].fx + pt[2].fx)/2.0;			my2 = (pt[1].fy + pt[2].fy)/2.0;
	if(dy1 < 1.0e-16 && dy2 < 1.0e-16) {
		cy = (pt[0].fy + pt[1].fy + pt[2].fy)/3.0;
		cx = (pt[0].fx + pt[1].fx + pt[2].fx)/3.0;
		r2 = 0.0;			return;
		}
	else if(dy1 < 1.0e-16) {
		cx = (pt[0].fx + pt[1].fx)/2.0;			cy = m2 * (cx - mx2) + my2;
		}
	else if(dy2 < 1.0e-16) {
		cx = (pt[2].fx + pt[1].fx)/2.0;			cy = m1 * (cx - mx1) + my1;
		}
	else {
		cx = (m1*mx1-m2*mx2+my2-my1)/(m1-m2);	cy = m1*(cx - mx1) + my1;
		}
	dx = pt[1].fx - cx;	dy = pt[1].fy - cy;		r2 = dx * dx + dy * dy;
}

bool
Triangle::TestVertex(double x, double y)
{
	double dx, dy;

	dx = x-cx;		dx = dx * dx;		dy = y-cy;		dy = dy * dy;
	return (dx+dy)<r2;
}

Triangulate::Triangulate(Triangle *t_list)
{
	trl = t_list;		edges = 0L;
}

bool
Triangulate::AddEdge(fPOINT3D *p1, fPOINT3D *p2)
{
	edge *ce, *ne;

	//if edge exists delete both the new and the existing edge
	for(ce = edges, ne = 0L; (ce); ) {
		if((ce->p1.fx == p1->fx && ce->p1.fy == p1->fy && ce->p1.fz == p1->fz
			&& ce->p2.fx == p2->fx && ce->p2.fy == p2->fy && ce->p2.fz == p2->fz)
			|| (ce->p2.fx == p1->fx && ce->p2.fy == p1->fy && ce->p2.fz == p1->fz
			&& ce->p1.fx == p2->fx && ce->p1.fy == p2->fy && ce->p1.fz == p2->fz)) {
			if(ne) ne->next = ce->next;
			else edges = ce->next;
			delete ce;					return true;
			}
		ne = ce;	ce = ce->next;
		}
	//come here for new edge
	if(ne = new edge()) {
		ne->p1.fx = p1->fx;		ne->p1.fy = p1->fy;		ne->p1.fz = p1->fz;
		ne->p2.fx = p2->fx;		ne->p2.fy = p2->fy;		ne->p2.fz = p2->fz;
		ne->next = edges;		edges = ne;
		}
	return false;
}

bool
Triangulate::AddVertex(fPOINT3D *v)
{
	Triangle *trc, *trn, *tr1;
	edge *ce, *ae;

	for(trc = trl, trn = 0L, edges = 0L; (trc);) {
		tr1 = trc->next;
		//delete triangles whose circumcircle enclose the new vertex
		if(trc->TestVertex(v->fx, v->fy)) {
			AddEdge(&trc->pt[0], &trc->pt[1]);		AddEdge(&trc->pt[1], &trc->pt[2]);
			AddEdge(&trc->pt[0], &trc->pt[2]);
			if(trn) trn->next = trc->next;
			else trl = trc->next;
			if(trl == trc) trl = 0L;	
			delete trc;
			}
		else trn = trc;
		trc = tr1;
		}
	//create new triangles from those edges which where found only once
	for(ce = edges; (ce); ) {
		if(trn = new Triangle()) {
			trn->pt[0].fx = ce->p1.fx;	trn->pt[0].fy = ce->p1.fy;	trn->pt[0].fz = ce->p1.fz;
			trn->pt[1].fx = ce->p2.fx;	trn->pt[1].fy = ce->p2.fy;	trn->pt[1].fz = ce->p2.fz;
			trn->pt[2].fx = v->fx;		trn->pt[2].fy = v->fy;		trn->pt[2].fz = v->fz;
			trn->SetRect();				trn->next = trl;			trl = trn;
			ae = ce->next;				delete(ce);					ce = ae;
			}
		}
	return true;
}

Triangle* Triangulate1(char *xr, char *yr, char *zr, DataObj *data)
{
	AccRange *rX, *rY, *rZ;
	int i, j, n, rx, cx, ry, cy, rz, cz;
	double zMin;
	fPOINT3D *da;
	fRECT lim;
	Triangle *trl, *trn;
	Triangulate *tria;

	rX = rY = rZ = 0L;				trl = trn  = 0L;
	if((rX = new AccRange(xr)) && (rY = new AccRange(yr)) && (rZ = new AccRange(zr))
		&& rX->GetFirst(&cx, &rx) && rY->GetFirst(&cy, &ry) && rZ->GetFirst(&cz, &rz)
		&& (n = rX->CountItems()) && (da = (fPOINT3D*)malloc(n * sizeof(fPOINT3D)))
		&& (trl = new Triangle()) && (trn = new Triangle())) {
		//get minima and maxima
		for(i = j = 0; i < n; i++) {
			if(rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) && rZ->GetNext(&cz, &rz)) {
				data->GetValue(rx, cx, &da[j].fx);	data->GetValue(ry, cy, &da[j].fy);
				data->GetValue(rz, cz, &da[j].fz);	j++;
				}
			}
		if(!j) {
			free(da); delete rX;	delete rY;	delete rZ;	return trl;
			}
		for(i = 0, j = n; i < n; i++) {
			if(i) {
				if(da[i].fx < lim.Xmin) lim.Xmin = da[i].fx;	if(da[i].fx > lim.Xmax) lim.Xmax = da[i].fx;
				if(da[i].fy < lim.Ymin) lim.Ymin = da[i].fy;	if(da[i].fy > lim.Ymax) lim.Ymax = da[i].fy;
				if(da[i].fz < zMin) zMin = da[i].fz;
				}
			else {
				lim.Xmax = lim.Xmin = da[i].fx;		lim.Ymax = lim.Ymin = da[i].fy;		zMin = da[i].fz;
				}
			}
		//setup two super triangles
		trl->pt[0].fz = trl->pt[1].fz = trl->pt[2].fz = zMin;
		trn->pt[0].fz = trn->pt[1].fz = trn->pt[2].fz = zMin;
		trl->pt[0].fx = trn->pt[0].fx = trl->pt[2].fx = lim.Xmin;
		trl->pt[0].fy = trn->pt[0].fy = trn->pt[1].fy = lim.Ymin;
		trl->pt[1].fx = trn->pt[2].fx = trn->pt[1].fx = lim.Xmax;
		trl->pt[1].fy = trn->pt[2].fy = trl->pt[2].fy = lim.Ymax;
		trl->SetRect();			trn->SetRect();
		trl->next = trn;		trn->next = 0L;
		//do triangulation
		tria = new Triangulate(trl);
		for(i = 0; i < n; i++) {
			tria->AddVertex(&da[i]);
			}
		free(da);
		}
	if(rX) delete rX;	if(rY) delete rY;	if(rZ) delete rZ;
	trl = tria->trl;	delete tria;		return trl;
}

Ribbon *SurfTria(GraphObj *parent, DataObj *data, char *xr, char *yr, char *zr)
{
	Triangle *trl, *trc, *trn;
	int i, j, n, npl;
	double tmp;
	Plane3D **planes;

	trl = Triangulate1(xr, zr, yr, data);
	for(i = 0, trc = trl; trc; i++) trc = trc->next;
	if((n = i) && (planes = (Plane3D**)malloc(n*sizeof(Plane3D*)))) 
		for(i = npl = 0, trc = trl; trc && i < n; i++) {
		for(j = 0; j < 4; j++) {	//swap y and z values;
			tmp = trc->pt[j].fz;	trc->pt[j].fz = trc->pt[j].fy;	trc->pt[j].fy = tmp;
			}
		planes[npl++] = new Plane3D(0L, data, trc->pt, 4);
		trn = trc->next;	delete trc;		trc = trn;
		}
	if(npl) return new Ribbon(parent, data, (GraphObj**)planes, npl);
	return 0L;
}
