/* FileAction.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.actions;

import java.awt.Window;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

/**
 * Common (abstract) super class of {@link FileOpenAction} and
 * {@link FileSaveAction}. Clients should consider extending one of those
 * classes instead of this one.
 */
public abstract class FileAction extends AbstractAction {

    /**
     * The file chooser which is used by this action.
     */
    protected JFileChooser fileChooser;
    
    /**
     * Parent window for any dialogs which need to be shown, in particular
     * the file chooser dialog.
     */
    protected Window parent;

    /**
     * Create an action of this type
     * @param parent Window which will be used as parent for
     * all dialogs
     * @param filters File filters for the file chooser dialog
     */
    public FileAction(Window parent, FileFilter... filters) {
        this.parent = parent;
        this.fileChooser = createFileChooser();
        if (filters.length == 0) {
            fileChooser.setAcceptAllFileFilterUsed(true);
        } else if (filters.length == 1) {
            this.fileChooser.setFileFilter(filters[0]);
        } else {
            for (FileFilter filter : filters) {
                this.fileChooser.addChoosableFileFilter(filter);
            }
        }
    }

    /**
     * Factory method which creates the file chooser at initialization time.
     * to initialize its 'approve' button.
     */
    protected JFileChooser createFileChooser() {
        return new JFileChooser();
    }

    /**
     * Called when the dialog is canceled. This implementation is empty.
     */
    protected void dialogCanceled() {
    }

    /**
     * Called when the dialog is dismissed. This implementation calls
     * {@link #dialogCanceled}.
     */
    protected void dialogClosed() {
        dialogCanceled();
    }

    /**
     * Called when the dialog is approved.
     */
    protected abstract void dialogApproved();

    /**
     * Shows the dialog appropriate for this type of file action 
     */
    protected abstract int showDialog();
    
    /**
     * Shows the file chooser dialog and dispatches to the
     * appropriate {@code dialog...} method depending on
     * the users choice
     */
    public void actionPerformed(ActionEvent e) {
        int result = showDialog();
        if (result == JFileChooser.APPROVE_OPTION) {
            dialogApproved();
        } else if (result == JFileChooser.CANCEL_OPTION) {
            dialogCanceled();
        } else {
            dialogClosed();
        }
    }
}
