/*!
	========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
*/
package com.sap.dbtech.jdbc.translators;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;

import com.sap.dbtech.jdbc.DBProcParameterInfo;
import com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB;
import com.sap.dbtech.jdbc.packet.DataPart;
import com.sap.dbtech.util.MessageKey;
import com.sap.dbtech.util.MessageTranslator;
import com.sap.dbtech.util.StructuredMem;

/**
 * Translator for OMS Streams.
 */
public class ABAPStreamTranslator extends DBTechTranslator {

	private static final int STREAMTYPE_ASCII      = 1;
	private static final int STREAMTYPE_UNICODE    = 2;
	private static final int STREAMTYPE_BYTE       = 3;
	private static final int STREAMTYPE_STRUCTURED = 4;
	
	private DBProcParameterInfo parameterStructure;
	private boolean             isUnicodeConnection;
	private int                 streamtype;
	   
	public ABAPStreamTranslator(int mode, int ioType, int dataType, int len, int ioLen, int bufpos, boolean readOnly, boolean autoIncre) {
		super(mode, ioType, dataType, len, ioLen, bufpos, readOnly, autoIncre);
		this.isUnicodeConnection = false;
		this.streamtype = 0;
	}

	public ABAPStreamTranslator(int mode, int ioType, 
								int dataType, int len, int ioLen, 
								int bufpos, boolean unicode, boolean readOnly, boolean autoIncre) {
		super(mode, ioType, dataType, len, ioLen, bufpos, readOnly, autoIncre);
		this.isUnicodeConnection = unicode;
		this.streamtype = 0;	
	}

	protected void putSpecific(DataPart dataPart, Object data) throws SQLException {
		ABAPStreamDescriptor d = (ABAPStreamDescriptor) data;
		d.putDescriptor(dataPart, this.bufpos);
	}

	protected Object transSpecificForInput(Object obj) throws SQLException {
		
		// TODO: translate specific objects
		
		// we have not to handle:
		// - a String
		// - a Byte
		// - a byte array
		// - a char array
		// we have to handle
		// - stream
		// - reader
		// - java.sql.Array
		// - java.sql.SQLData
		// - java.sql.Struct[]
		// - java.lang.Object[][] 	
		
		return null;
	}


	public Object transBinaryStreamForInput(InputStream stream, int length)
		throws SQLException {
			
		if(this.streamtype == STREAMTYPE_BYTE) {
			if(stream == null) {
				return null;
			}
			return new BinaryABAPStreamPutval(this, stream, length);
		} else {
			throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_CONVERSION_BYTESTREAM));
		}
	}
	
	public Object transStringForInput(String val)
		throws SQLException	{
		if (val == null) {
			return null;
		} else {
			switch(this.streamtype) {
			case STREAMTYPE_ASCII: {
				if(isUnicodeConnection) {
					return new AsciiABAPStreamPutval(this, com.sap.dbtech.util.StringUtil.bytes_ascii7bit(val), true);	
				} else {
					return new AsciiABAPStreamPutval(this, com.sap.dbtech.util.StringUtil.bytes_iso8859_1(val), false);
				}
			}
			case STREAMTYPE_UNICODE:
                if(isUnicodeConnection) {
                    return new UnicodeABAPStreamPutval(this, val.toCharArray());	
                } else {
                    return new AsciiABAPStreamPutval(this, com.sap.dbtech.util.StringUtil.bytes_ascii7bit(val), false);	
                }
			default:
				throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_CONVERSION_STRINGSTREAM));
			}
		}
	}
	
	public Object transCharacterStreamForInput (java.io.Reader reader, int length) 
		throws SQLException {
		switch(this.streamtype) {
        case STREAMTYPE_ASCII: {
            if(reader == null) {
                return null;
            } else {
                if(isUnicodeConnection) {
                    return new AsciiABAPStreamPutval(this, new ReaderInputStream(reader, true), length, true);
                } else  {
                    return new AsciiABAPStreamPutval(this, new ReaderInputStream(reader, false), length, false);
                }
            }
        }
        case STREAMTYPE_UNICODE:
            if(isUnicodeConnection) {
                return new UnicodeABAPStreamPutval(this, reader, length);
            } else {
                return new AsciiABAPStreamPutval(this, new ReaderInputStream(reader, true), length, false);
            }
        default:
            throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_CONVERSION_STRINGSTREAM));
		}
	}
    
	public void setProcParamInfo(DBProcParameterInfo info) {
		this.parameterStructure = info;	
		if(this.parameterStructure != null) {
			if(this.parameterStructure.getMemberCount() == 1) {
				DBProcParameterInfo.StructureElement s = this.parameterStructure.getMember(0);
				if(s.typeName.equals("CHAR")) {
					if(s.codeType.equals("ASCII")) {
						this.streamtype = STREAMTYPE_ASCII;
						return; 
					} else if(s.codeType.equals("BYTE")) {
						this.streamtype = STREAMTYPE_BYTE;
						return;
					}
				} else if(s.typeName.equals("WYDE")) {
					this.streamtype = STREAMTYPE_UNICODE;
					return;		
				}
			} 
			this.streamtype = STREAMTYPE_STRUCTURED;
		}
	}
	
	public boolean isStreamKind() {
		return true;
	}

	public Object transClobForInput(Clob clob) throws SQLException {
		if(this.streamtype == STREAMTYPE_ASCII || this.streamtype==STREAMTYPE_UNICODE) {
		 	if(clob == null) {
		 		return null;
		 	}
		 	Reader r = clob.getCharacterStream();
		 	StringBuffer data = new StringBuffer();
		 	char[] readerData = new char[4096];
		 	int charsRead = 0;
		 	do {
		 		try { 
		 			charsRead = r.read(readerData);
		 			if(charsRead != -1) {
		 				data.append(readerData, 0, charsRead);
		 			}
		 		} catch(IOException ioex) {
		 			throw new SQLExceptionSapDB(MessageKey.ERROR_STREAM_SOURCEREAD, ioex.getMessage());
		 		}
		 	} while(charsRead != -1);
		 	try { 
		 		r.close();
		 	} catch(IOException ignoredIoEx) {
		 		// Ignore this.
		 	}
		 	return transStringForInput(data.toString());
		} else {
			throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_CONVERSION_STRINGSTREAM));			
		}
	}

	public Object transBytesForInput(byte[] val) throws SQLException {
		if(val == null) {
			return transBinaryStreamForInput(null, -1);
		} else {
			return transBinaryStreamForInput(new ByteArrayInputStream(val), -1);
		}
	}
	
	public AbstractABAPStreamGetval createGetval() 
		throws SQLException {
		switch(this.streamtype) {
		case STREAMTYPE_ASCII:
			return new AsciiABAPStreamGetval(this);
		case STREAMTYPE_UNICODE:
			if(isUnicodeConnection) {
				return new UnicodeABAPStreamGetval(this);
			} else {
				return new AsciiABAPStreamGetval(this);		
			}
		case STREAMTYPE_BYTE:
			return new BinaryABAPStreamGetval(this);
		case STREAMTYPE_STRUCTURED:
			return new StructureABAPStreamGetval(this,this.parameterStructure, this.isUnicodeConnection);
		default:
			throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STREAM_UNKNOWNTYPE));
		}
	}

	/**
	 * Looks up the GETVAL instance in the parameter controller.
	 * @param controller The parameter controller to query.
	 * @return The GETVAL instance.
	 * @throws SQLException if no GETVAL instance is found.
	 */
	private AbstractABAPStreamGetval getGetval(SQLParamController controller) 
		throws SQLException {
		AbstractABAPStreamGetval gv = controller.getOMSGetval(this.getColIndex());
		if(gv == null) {
			throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STREAM_NODATA));
		} else {
			return gv;
		}			
	}

	
	public String getString(SQLParamController controller, StructuredMem mem)
		throws SQLException {			
			return getGetval(controller).getString();
	}

	
	public InputStream getAsciiStream(SQLParamController controller,
									StructuredMem mem,
									StructuredMem longData)
		throws SQLException {
		return getGetval(controller).getAsciiStream();
	}

	public BigDecimal getBigDecimal(
		int scale,
		SQLParamController controller,
		StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getBigDecimal(scale);
	}

	public BigDecimal getBigDecimal(
		SQLParamController controller,
		StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getBigDecimal();
	}

	public InputStream getBinaryStream(
		SQLParamController controller,
		StructuredMem mem,
		StructuredMem longData)
		throws SQLException {
		return getGetval(controller).getBinaryStream();			
	}

	public Blob getBlob(
		SQLParamController controller,
		StructuredMem mem,
		StructuredMem longData)
		throws SQLException {
		return getGetval(controller).getBlob();		
	}

	public boolean getBoolean(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getBoolean();
	}

	public byte getByte(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getByte();
	}

	public byte[] getBytes(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getBytes();
	}

	public Reader getCharacterStream(
		SQLParamController controller,
		StructuredMem mem,
		StructuredMem longData)
		throws SQLException {
		return getGetval(controller).getCharacterStream();
	}

	public Clob getClob(
		SQLParamController controller,
		StructuredMem mem,
		StructuredMem longData)
		throws SQLException {
		return getGetval(controller).getClob();
	}

	public Date getDate(
		SQLParamController controller,
		StructuredMem mem,
		Calendar cal)
		throws SQLException {
		return getGetval(controller).getDate(cal);
	}

	public double getDouble(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getDouble();				
	}

	/* (non-Javadoc)
	 * @see com.sap.dbtech.jdbc.translators.DBTechTranslator#getFloat(com.sap.dbtech.jdbc.translators.SQLParamController, com.sap.dbtech.util.StructuredMem)
	 */
	public float getFloat(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getFloat();
	}

	public int getInt(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getInt();
	}

	public long getLong(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getLong();
	}

	
	public Object getObject(SQLParamController controller, StructuredMem mem)
		throws SQLException {
		return getGetval(controller).getObject();
	}

	public Time getTime(
		SQLParamController controller,
		StructuredMem mem,
		Calendar cal)
		throws SQLException {
		return getGetval(controller).getTime(cal);
	}

	public Timestamp getTimestamp(
		SQLParamController controller,
		StructuredMem mem,
		Calendar cal)
		throws SQLException {
		return getGetval(controller).getTimestamp(cal);
	}
	
	public String getBaseTypeName() {
		return this.parameterStructure.getBaseTypeName();
	}

	
	public String getColumnTypeName() {
		return this.parameterStructure.getSQLTypeName();
	}

}
