/**
 * ========================================
 * JFreeReport : a free Java report library
 * ========================================
 *
 * Project Info:  http://www.object-refinery.com/jfreereport/index.html
 * Project Lead:  Thomas Morgner (taquera@sherito.org);
 *
 * (C) Copyright 2000-2003, by Simba Management Limited and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * ------------------------------
 * ParseException.java
 * ------------------------------
 * (C)opyright 2003, by Thomas Morgner and Contributors.
 *
 * Original Author:  Thomas Morgner;
 * Contributor(s):   David Gilbert (for Simba Management Limited);
 *
 * $Id: ParseException.java,v 1.3 2003/06/13 15:46:58 mungady Exp $
 *
 * Changes
 * -------------------------
 * 10.06.2003 : Initial version
 *
 */

package org.jfree.xml;

import org.xml.sax.Locator;
import org.xml.sax.SAXException;

public class ParseException extends SAXException {

    /** The line, where the error occured. */
    private int line;

    /** The column, where the error occured. */
    private int column;

    /**
     * Creates a new ParseException with the given message.
     *
     * @param message the message
     */
    public ParseException(String message) {
        super(message);
        fillLocation(null);
    }

    /**
     * Creates a new ParseException with the given root exception.
     *
     * @param e the exception
     */
    public ParseException(Exception e) {
        super(e);
        fillLocation(null);
    }

    /**
     * Creates a new ParseException with the given message and root exception.
     *
     * @param s the message
     * @param e the exception
     */
    public ParseException(String s, Exception e) {
        super(s, e);
        fillLocation(null);
    }

    /**
     * Creates a new ParseException with the given message and the locator.
     *
     * @param message the message
     * @param locator the locator of the parser
     */
    public ParseException(String message, Locator locator) {
        super(message);
        fillLocation(locator);
    }

    /**
     * Creates a new ParseException with the given root exception
     * and the locator.
     *
     * @param e the exception
     * @param locator the locator of the parser
     */
    public ParseException(Exception e, Locator locator) {
        super(e);
        fillLocation(locator);
    }

    /**
     * Creates a new ParseException with the given message, root exception
     * and the locator.
     *
     * @param s the message
     * @param e the exception
     * @param locator the locator of the parser
     */
    public ParseException(String s, Exception e, Locator locator) {
        super(s, e);
        fillLocation(locator);
    }

    /**
     * Modifies the message to give more detailed location information.
     *
     * @return the modified exception message.
     */
    public String getMessage() {
        StringBuffer message = new StringBuffer(super.getMessage());
        message.append(" [Location: Line=");
        message.append(line);
        message.append(" Column=");
        message.append(column);
        message.append("] ");
        return message.toString();
    }

    /**
     * Fills the location with the given locator.
     *
     * @param locator the locator or null.
     */
    protected void fillLocation (Locator locator)
    {
        if (locator == null)
        {
            line = -1;
            column = -1;
        }
        else
        {
            line = locator.getLineNumber();
            column = locator.getColumnNumber();
        }
    }

    /**
     * Returns the line of the parse position where the error occured.
     *
     * @return the line number or -1 if not known.
     */
    public int getLine() {
        return line;
    }

    /**
     * Returns the column of the parse position where the error occured.
     *
     * @return the column number or -1 if not known.
     */
    public int getColumn() {
        return column;
    }
}
